package org.mule.tools.cxf.utils.policy;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.Validate;
import org.apache.cxf.Bus;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.endpoint.EndpointImpl;
import org.apache.cxf.frontend.ClientProxy;
import org.apache.cxf.ws.policy.AbstractPolicyInterceptorProvider;
import org.apache.cxf.ws.policy.PolicyInterceptorProvider;
import org.apache.cxf.ws.policy.PolicyInterceptorProviderRegistry;
import org.apache.cxf.ws.security.sts.provider.operation.ValidateOperation;

/**
 * This class knows how to registers CXF {@link PolicyInterceptorProvider} in a
 * {@link Client}. Which are needed by CXF 2.6.0 when custom policy are defined
 * in the service definition.
 * 
 * Usually you'll notice them cause they have different name spaces.
 * 
 * 
 * @author damiansima
 * 
 */
public class PolicyRegistrator {

    private List<PolicyInterceptorProvider> policyProviders = new ArrayList<PolicyInterceptorProvider>();

    /**
     * Add a {@link PolicyInterceptorProvider} to the internal structure for
     * future registration in a given client
     * 
     * @param policyPorvider
     */
    public void addPolicyProvider(PolicyInterceptorProvider policyPorvider) {
        policyProviders.add(policyPorvider);
    }

    /**
     * Add all a {@link PolicyInterceptorProvider} to the internal structure for
     * future registration in a given client
     * 
     * @param policyProvidersList
     */
    public void addAllPolicyProviders(List<PolicyInterceptorProvider> policyProvidersList) {
        policyProviders.addAll(policyProvidersList);
    }

    public List<PolicyInterceptorProvider> getPolicyProviders() {
        return policyProviders;
    }

    /**
     * Clear all {@link PolicyInterceptorProvider} from the internal structure
     */
    public void clearPolicyProviders() {
        policyProviders.clear();
    }

    /**
     * Register all the {@link PolicyInterceptorProvider} in the servicePort
     * provided
     * 
     * @param servicePort
     * @return the client with the {@link PolicyInterceptorProvider} registered
     *         in it
     */
    public Client registerPolicyProviders(Object servicePort) {
        Validate.notNull(servicePort, "The service port should not be null");

        Client client = ClientProxy.getClient(servicePort);

        return registerPolicyProviders(client);
    }

    /**
     * Register all the {@link PolicyInterceptorProvider} in the {@link Client}
     * provided
     * 
     * @param client
     * @return the client with the {@link PolicyInterceptorProvider} registered
     *         in it
     */
    public Client registerPolicyProviders(Client client) {
        Validate.notEmpty(policyProviders, "There are no policy providers to register, please add them");
        Validate.notNull(client, "The client port should not be null");

        // Register Policy Providers
        Bus bus = ((EndpointImpl) client.getEndpoint()).getBus();
        PolicyInterceptorProviderRegistry pipr = bus.getExtension(PolicyInterceptorProviderRegistry.class);

        if (pipr != null) {
            // this is for CXF 2.6.x
            for (PolicyInterceptorProvider policyProvider : policyProviders) {
                pipr.register(policyProvider);
            }

        } else {
            // this is for CXF 2.5.x
            // place warning
        }

        return client;

    }
}
