/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package sun.security.mule.jgss.krb5;

import java.io.IOException;
import java.io.InputStream;

import org.ietf.jgss.GSSException;

import sun.security.mule.krb5.Config;
import sun.security.mule.krb5.Credentials;
import sun.security.mule.krb5.EncryptionKey;
import sun.security.mule.krb5.KrbApRep;
import sun.security.mule.krb5.KrbApReq;
import sun.security.mule.krb5.KrbException;

class AcceptSecContextToken extends InitialToken {

	private KrbApRep apRep = null;

	/**
	 * Creates an AcceptSecContextToken for the context acceptor to send to
	 * the context initiator.
	 */
	public AcceptSecContextToken(Krb5Context context, KrbApReq apReq, Config kerberosConfig) throws KrbException, IOException {

		/*
		 * RFC 1964, section 1.2 states:
		 * (1) context key: uses Kerberos session key (or subkey, if
		 * present in authenticator emitted by context initiator) directly
		 * 
		 * This does not mention context acceptor. Hence we will not
		 * generate a subkey on the acceptor side. Note: Our initiator will
		 * still allow another acceptor to generate a subkey, even though
		 * our acceptor does not do so.
		 */
		boolean useSubkey = false;

		boolean useSequenceNumber = true;

		apRep = new KrbApRep(apReq, useSequenceNumber, useSubkey, kerberosConfig);

		context.resetMySequenceNumber(apRep.getSeqNumber().intValue());

		/*
		 * Note: The acceptor side context key was set when the
		 * InitSecContextToken was received.
		 */
	}

	/**
	 * Creates an AcceptSecContextToken at the context initiator's side
	 * using the bytes received from the acceptor.
	 */
	public AcceptSecContextToken(Krb5Context context, Credentials serviceCreds, KrbApReq apReq, InputStream is, Config kerberosConfig) throws IOException,
			GSSException, KrbException {

		int tokenId = ((is.read() << 8) | is.read());

		if (tokenId != Krb5Token.AP_REP_ID)
			throw new GSSException(GSSException.DEFECTIVE_TOKEN, -1, "AP_REP token id does not match!");

		byte[] apRepBytes = new sun.security.util.DerValue(is).toByteArray();

		KrbApRep apRep = new KrbApRep(apRepBytes, serviceCreds, apReq, kerberosConfig);

		/*
		 * Allow the context acceptor to set a subkey if desired, even
		 * though our context acceptor will not do so.
		 */
		EncryptionKey subKey = apRep.getSubKey();
		if (subKey != null) {
			context.setKey(subKey);
			/*
			 * System.out.println("\n\nSub-Session key from AP-REP is: " +
			 * getHexBytes(subKey.getBytes()) + "\n");
			 */
		}

		Integer apRepSeqNumber = apRep.getSeqNumber();
		int peerSeqNumber = (apRepSeqNumber != null ? apRepSeqNumber.intValue() : 0);
		context.resetPeerSequenceNumber(peerSeqNumber);
	}

	public final byte[] encode() throws IOException {
		byte[] apRepBytes = apRep.getMessage();
		byte[] retVal = new byte[2 + apRepBytes.length];
		writeInt(Krb5Token.AP_REP_ID, retVal, 0);
		System.arraycopy(apRepBytes, 0, retVal, 2, apRepBytes.length);
		return retVal;
	}
}
