/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package sun.security.mule.jgss.krb5;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.security.auth.DestroyFailedException;
import javax.security.auth.Subject;
import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosTicket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * This utility looks through the current Subject and retrieves a ticket or key
 * for the desired client/server principals.
 * 
 * @author Ram Marti
 * @since 1.4.2
 */

class SubjectComber {

//	private static final boolean DEBUG = Krb5Util.DEBUG;
	static final private Log logger = LogFactory.getLog(SubjectComber.class);

	/**
	 * Default constructor
	 */
	private SubjectComber() { // Cannot create one of these
	}

	static Object find(Subject subject, String serverPrincipal, String clientPrincipal, Class credClass) {

		return findAux(subject, serverPrincipal, clientPrincipal, credClass, true);
	}

	static Object findMany(Subject subject, String serverPrincipal, String clientPrincipal, Class credClass) {

		return findAux(subject, serverPrincipal, clientPrincipal, credClass, false);
	}

	/**
	 * Find the ticket or key for the specified client/server principals
	 * in the subject. Returns null if the subject is null.
	 * 
	 * @return the ticket or key
	 */
	private static Object findAux(Subject subject, String serverPrincipal, String clientPrincipal, Class credClass, boolean oneOnly) {

		if (subject == null) {
			return null;
		} else {
			List<Object> answer = (oneOnly ? null : new ArrayList<Object>());

			if (credClass == KerberosKey.class) {
				// We are looking for a KerberosKey credentials for the
				// serverPrincipal
				Iterator<KerberosKey> iterator = subject.getPrivateCredentials(KerberosKey.class).iterator();
				while (iterator.hasNext()) {
					KerberosKey key = iterator.next();
					if (serverPrincipal == null || serverPrincipal.equals(key.getPrincipal().getName())) {
						if (logger.isDebugEnabled()) {
							logger.debug("Found key for " + key.getPrincipal() + "(" + key.getKeyType() + ")");
						}
						if (oneOnly) {
							return key;
						} else {
							if (serverPrincipal == null) {
								// Record name so that keys returned will all
								// belong to the same principal
								serverPrincipal = key.getPrincipal().getName();
							}
							answer.add(key);
						}
					}
				}
			} else if (credClass == KerberosTicket.class) {
				// we are looking for a KerberosTicket credentials
				// for client-service principal pair
				Set<Object> pcs = subject.getPrivateCredentials();
				synchronized (pcs) {
					Iterator<Object> iterator = pcs.iterator();
					while (iterator.hasNext()) {
						Object obj = iterator.next();
						if (obj instanceof KerberosTicket) {
							KerberosTicket ticket = (KerberosTicket) obj;
							if (logger.isDebugEnabled()) {
								logger.debug("Found ticket for " + ticket.getClient() + " to go to " + ticket.getServer()
										+ " expiring on " + ticket.getEndTime());
							}
							if (!ticket.isCurrent()) {
								// let us remove the ticket from the Subject
								// Note that both TGT and service ticket will be
								// removed upon expiration
								if (!subject.isReadOnly()) {
									iterator.remove();
									try {
										ticket.destroy();
										if (logger.isDebugEnabled()) {
											logger.debug("Removed and destroyed " + "the expired Ticket \n" + ticket);

										}
									} catch (DestroyFailedException dfe) {
										if (logger.isDebugEnabled()) {
											logger.debug("Expired ticket not" + " detroyed successfully. " + dfe);
										}
									}

								}
							} else {
								if (serverPrincipal == null || ticket.getServer().getName().equals(serverPrincipal)) {

									if (clientPrincipal == null || clientPrincipal.equals(ticket.getClient().getName())) {
										if (oneOnly) {
											return ticket;
										} else {
											// Record names so that tickets will
											// all belong to same principals
											if (clientPrincipal == null) {
												clientPrincipal = ticket.getClient().getName();
											}
											if (serverPrincipal == null) {
												serverPrincipal = ticket.getServer().getName();
											}
											answer.add(ticket);
										}
									}
								}
							}
						}
					}
				}
			}
			return answer;
		}
	}
}
