/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package sun.security.mule.krb5.cxf;

import java.net.URL;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.common.util.Base64Utility;
import org.apache.cxf.configuration.security.AuthorizationPolicy;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Message;
import org.apache.cxf.transport.http.auth.HttpAuthHeader;
import org.apache.cxf.transport.http.auth.HttpAuthSupplier;
import org.apache.cxf.ws.security.SecurityConstants;
import org.apache.ws.security.WSSecurityException;
import org.apache.ws.security.spnego.SpnegoClientAction;
import org.ietf.jgss.GSSException;

import sun.security.mule.krb5.Config;

public class MuleSpnegoAuthSupplier implements HttpAuthSupplier {
	
	static final private Log LOG = LogFactory.getLog(MuleSpnegoAuthSupplier.class);
	
	private byte[] token;
	private SpnegoTokenContext spnegoTokenContext;
	
	/**
	 * Can be set on the jaxws:properties. If set to true then the kerberos oid is used
	 * instead of the default spnego OID
	 */

	public boolean requiresRequestCaching() {
		return false;
	}

	public String getAuthorization(AuthorizationPolicy authPolicy, URL currentURL, Message message, String fullHeader) {
		if (!HttpAuthHeader.AUTH_TYPE_NEGOTIATE.equals(authPolicy.getAuthorizationType())) {
			return null;
		}

		String spn = "HTTP/" + currentURL.getHost();
		LOG.debug("Adding authorization service ticket for service principal name: " + spn);

		token = issueToken(message);

		return HttpAuthHeader.AUTH_TYPE_NEGOTIATE + " " + Base64Utility.encode(token);
	}

	
	private byte[] issueToken(Message message) {
		Config kerberosConfig = (Config) message.get(KerberosConstants.KERBEROS_CONFIG);
		
		if (kerberosConfig == null) {
			String msg = "Cannot find a Kerberos config in the request of the message. Key:" + KerberosConstants.KERBEROS_CONFIG;
			if (LOG.isErrorEnabled()) {
				LOG.error(msg);
			}
			throw new RuntimeException(msg);
		}
		
		if (kerberosConfig.getKerberosContext() == null) {
			String kerberosSpn = (String) message.getContextualProperty(SecurityConstants.KERBEROS_SPN);
			CallbackHandler callbackHandler = (CallbackHandler) message.getContextualProperty(SecurityConstants.CALLBACK_HANDLER);
	
			spnegoTokenContext = spnegoTokenContext != null ? spnegoTokenContext : new SpnegoTokenContext();
			Object spnegoClientAction = message.getContextualProperty(SecurityConstants.SPNEGO_CLIENT_ACTION);
			if (spnegoClientAction instanceof SpnegoClientAction) {
				spnegoTokenContext.setSpnegoClientAction((SpnegoClientAction) spnegoClientAction);
			}
			
			spnegoTokenContext.setKerberosConfig(kerberosConfig);
	
			try {
				spnegoTokenContext.retrieveServiceTicket(callbackHandler, kerberosSpn);
			} catch (WSSecurityException e) {
				throw new Fault(e);
			}
			
			return spnegoTokenContext.getToken();
		} else {
			/*
			 * Once the first calls has been made (AS_REQ, TGS_REQ)
			 * the Krb5Context stores itself in the kerberosConfiguration.
			 * We need to reset the states in order to being able to reuse it.
			 * 
			 * Now that the context has a TGT and a TGS it will only renew the Authenticator in the token and generete the new one
			 * preventing from redo the authentication process again
			 */
			byte[] token = new byte[0];
			try {
				kerberosConfig.getKerberosContext().resetStateToInProgress();
				return kerberosConfig.getKerberosContext().initSecContext(token, 0, token.length);
			} catch (GSSException e) {
				throw new RuntimeException("cannot generate new AP_REQ", e);
			}
		}
	}	
}
