/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package sun.security.mule.krb5.cxf;

import java.security.Principal;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ws.security.WSSecurityException;
import org.apache.ws.security.spnego.DefaultSpnegoClientAction;
import org.apache.ws.security.spnego.DefaultSpnegoServiceAction;
import org.apache.ws.security.spnego.SpnegoClientAction;
import org.apache.ws.security.spnego.SpnegoServiceAction;
import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.MessageProp;
import org.mule.tools.cxf.utils.security.XRMSpnegoClientAction;

import sun.security.mule.jgss.krb5.Krb5NameElement;
import sun.security.mule.krb5.Config;
import sun.security.mule.krb5.login.Kerberos5LoginModule;


public class SpnegoTokenContext {

	private static final Log LOG = LogFactory.getLog(SpnegoTokenContext.class);

	private GSSContext secContext;
	private byte[] token;
	private boolean mutualAuth;
	private SpnegoClientAction clientAction = new DefaultSpnegoClientAction();
	private SpnegoServiceAction serviceAction = new DefaultSpnegoServiceAction();
	private Config kerberosConfig;

	
	public void setKerberosConfig(Config kerberosConfig) {
		this.kerberosConfig = kerberosConfig;
	}
	
	/**
	 * Retrieve a service ticket from a KDC using the Kerberos JAAS module, and set it in this
	 * BinarySecurityToken.
	 * 
	 * @param jaasLoginModuleName
	 *            the JAAS Login Module name to use
	 * @param callbackHandler
	 *            a CallbackHandler instance to retrieve a password (optional)
	 * @param serviceName
	 *            the desired Kerberized service
	 * @throws WSSecurityException
	 */
	public void retrieveServiceTicket(CallbackHandler callbackHandler, String serviceName)
			throws WSSecurityException {
		
		if (kerberosConfig == null) {
			String msg = "Trying to retrieve a service ticket without having suplied a Kerberos Configuration";
			if (LOG.isDebugEnabled()) LOG.debug(msg);
			throw new WSSecurityException(msg);
		}
		
		/*
		 * Using directly the Kerberos5LoginModule instead the LoginContext we skip the requirement of having a JAAS configuration
		 * file and the property settings.
		 * 
		 * We pass directly a hardcoded configuration of the module. The only thing it needs for now is enabling the debug mod
		 */
		
		Kerberos5LoginModule krb5Login = new Kerberos5LoginModule();
		
		Subject subject = new Subject();
        Map<String, Object> sharedState = new HashMap<String, Object>();
        Map<String, Object> options = new HashMap<String, Object>();
        
        // We pass to the Kerberos Login Module the kerberos config corresponding to this context
        krb5Login.initialize(subject, callbackHandler, sharedState, options, kerberosConfig);
		
        try {
			krb5Login.login();
			krb5Login.commit();
		} catch (LoginException ex) {
			if (LOG.isDebugEnabled()) {
				LOG.debug(ex.getMessage(), ex);
			}
			throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosLoginError", new Object[] { ex.getMessage() });
		}
        
        if (LOG.isDebugEnabled()) {
			LOG.debug("Successfully authenticated to the TGT");
		}
		
		Subject clientSubject = subject;
		
		Set<Principal> clientPrincipals = clientSubject.getPrincipals();
		if (clientPrincipals.isEmpty()) {
			throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosLoginError",
					new Object[] { "No Client principals found after login" });
		}
		
		Krb5NameElement krb5Name = null;
		try {
			krb5Name = Krb5NameElement.getInstance(serviceName, null, kerberosConfig);
		} catch (GSSException e) {
			throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosServiceTicketError - Cannot create the Krb5NameElement");
		}
		
		String principalName = krb5Name.getKrb5PrincipalName().getName();
		
		kerberosConfig.storeSubject(principalName, clientSubject);
		
		// Get the service ticket
		XRMSpnegoClientAction myClientAction = new XRMSpnegoClientAction(kerberosConfig);
		myClientAction.setServiceName(serviceName);
		myClientAction.setMutualAuth(mutualAuth);
		
		token = myClientAction.run();		
		
		if (token == null) {
			throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosServiceTicketError");
		}

		secContext = myClientAction.getContext();
		if (LOG.isDebugEnabled()) {
			LOG.debug("Successfully retrieved a service ticket");
		}

	}

	/**
	 * Validate a service ticket.
	 * 
	 * @param jaasLoginModuleName
	 * @param callbackHandler
	 * @param serviceName
	 * @param ticket
	 * @throws WSSecurityException
	 */
	public void validateServiceTicket(String jaasLoginModuleName, CallbackHandler callbackHandler, String serviceName, byte[] ticket)
			throws WSSecurityException {
		// Get a TGT from the KDC using JAAS
		LoginContext loginContext = null;
		try {
			if (callbackHandler == null) {
				loginContext = new LoginContext(jaasLoginModuleName);
			} else {
				loginContext = new LoginContext(jaasLoginModuleName, callbackHandler);
			}
			loginContext.login();
		} catch (LoginException ex) {
			if (LOG.isDebugEnabled()) {
				LOG.debug(ex.getMessage(), ex);
			}
			throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosLoginError", new Object[] { ex.getMessage() });
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("Successfully authenticated to the TGT");
		}

		// Get the service name to use - fall back on the principal
		Subject subject = loginContext.getSubject();
		String service = serviceName;
		if (service == null) {
			Set<Principal> principals = subject.getPrincipals();
			if (principals.isEmpty()) {
				throw new WSSecurityException(WSSecurityException.FAILURE, "kerberosLoginError",
						new Object[] { "No Client principals found after login" });
			}
			service = principals.iterator().next().getName();
		}

		// Validate the ticket
		serviceAction.setTicket(ticket);
		serviceAction.setServiceName(service);
		token = (byte[]) Subject.doAs(subject, serviceAction);

		secContext = serviceAction.getContext();
		if (LOG.isDebugEnabled()) {
			LOG.debug("Successfully validated a service ticket");
		}

	}

	/**
	 * Whether to enable mutual authentication or not. This only applies to retrieve service ticket.
	 */
	public void setMutualAuth(boolean mutualAuthentication) {
		mutualAuth = mutualAuthentication;
	}

	/**
	 * Get the SPNEGO token that was created.
	 */
	public byte[] getToken() {
		return token;
	}

	/**
	 * Whether a connection has been established (at the service side)
	 */
	public boolean isEstablished() {
		if (secContext == null) {
			return false;
		}
		return secContext.isEstablished();
	}

	/**
	 * Unwrap a key
	 */
	public byte[] unwrapKey(byte[] secret) throws WSSecurityException {
		MessageProp mProp = new MessageProp(0, true);
		try {
			return secContext.unwrap(secret, 0, secret.length, mProp);
		} catch (GSSException e) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Error in cleaning up a GSS context", e);
			}
			throw new WSSecurityException(WSSecurityException.FAILURE, "spnegoKeyError");
		}
	}

	/**
	 * Wrap a key
	 */
	public byte[] wrapKey(byte[] secret) throws WSSecurityException {
		MessageProp mProp = new MessageProp(0, true);
		try {
			return secContext.wrap(secret, 0, secret.length, mProp);
		} catch (GSSException e) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Error in cleaning up a GSS context", e);
			}
			throw new WSSecurityException(WSSecurityException.FAILURE, "spnegoKeyError");
		}
	}

	/**
	 * Set a custom SpnegoClientAction implementation to use
	 */
	public void setSpnegoClientAction(SpnegoClientAction spnegoClientAction) {
		this.clientAction = spnegoClientAction;
	}

	/**
	 * Set a custom SpnegoServiceAction implementation to use
	 */
	public void setSpnegoServiceAction(SpnegoServiceAction spnegoServiceAction) {
		this.serviceAction = spnegoServiceAction;
	}

	public void clear() {
		token = null;
		mutualAuth = false;
		kerberosConfig = null;
		try {
			secContext.dispose();
		} catch (GSSException e) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Error in cleaning up a GSS context", e);
			}
		}
	}

}
