/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

/*
 * (C) Copyright IBM Corp. 1999 All Rights Reserved.
 * Copyright 1997 The Open Group Research Institute. All rights reserved.
 */

package sun.security.mule.krb5.internal;

import java.io.IOException;
import java.util.Vector;

import sun.security.mule.krb5.Asn1Exception;
import sun.security.mule.krb5.Config;
import sun.security.util.DerInputStream;
import sun.security.util.DerOutputStream;
import sun.security.util.DerValue;

/**
 * Implements the ASN.1 LastReq type.
 * 
 * <xmp>
 * LastReq ::= SEQUENCE OF SEQUENCE {
 * lr-type [0] Int32,
 * lr-value [1] KerberosTime
 * }
 * </xmp>
 * 
 * <p>
 * This definition reflects the Network Working Group RFC 4120 specification available at <a
 * href="http://www.ietf.org/rfc/rfc4120.txt"> http://www.ietf.org/rfc/rfc4120.txt</a>.
 */

public class LastReq {
	private LastReqEntry[] entry = null;
	private Config kerberosConfig;

	public LastReq(LastReqEntry[] entries, Config kerberosConfig) throws IOException {
		this.kerberosConfig = kerberosConfig;
		if (entries != null) {
			entry = new LastReqEntry[entries.length];
			for (int i = 0; i < entries.length; i++) {
				if (entries[i] == null) {
					throw new IOException("Cannot create a LastReqEntry");
				} else {
					entry[i] = (LastReqEntry) entries[i].clone();
				}
			}
		}

	}

	/**
	 * Constructs a LastReq object.
	 * 
	 * @param encoding
	 *            a Der-encoded data.
	 * @exception Asn1Exception
	 *                if an error occurs while decoding an ASN1 encoded data.
	 * @exception IOException
	 *                if an I/O error occurs while reading encoded data.
	 */

	public LastReq(DerValue encoding, Config kerberosConfig) throws Asn1Exception, IOException {
		this.kerberosConfig = kerberosConfig;
		Vector<LastReqEntry> v = new Vector<LastReqEntry>();
		if (encoding.getTag() != DerValue.tag_Sequence) {
			throw new Asn1Exception(Krb5.ASN1_BAD_ID);
		}
		while (encoding.getData().available() > 0) {
			v.addElement(new LastReqEntry(encoding.getData().getDerValue(), kerberosConfig));
		}
		if (v.size() > 0) {
			entry = new LastReqEntry[v.size()];
			v.copyInto(entry);
		}
	}

	/**
	 * Encodes an LastReq object.
	 * 
	 * @return the byte array of encoded LastReq object.
	 * @exception Asn1Exception
	 *                if an error occurs while decoding an ASN1 encoded data.
	 * @exception IOException
	 *                if an I/O error occurs while reading encoded data.
	 */
	public byte[] asn1Encode() throws Asn1Exception, IOException {
		DerOutputStream bytes = new DerOutputStream();
		if (entry != null && entry.length > 0) {
			DerOutputStream temp = new DerOutputStream();
			for (int i = 0; i < entry.length; i++)
				temp.write(entry[i].asn1Encode());
			bytes.write(DerValue.tag_Sequence, temp);
			return bytes.toByteArray();
		}
		return null;
	}

	/**
	 * Parse (unmarshal) a last request from a DER input stream. This form
	 * parsing might be used when expanding a value which is part of
	 * a constructed sequence and uses explicitly tagged type.
	 * 
	 * @exception Asn1Exception
	 *                on error.
	 * @param data
	 *            the Der input stream value, which contains one or more marshaled value.
	 * @param explicitTag
	 *            tag number.
	 * @param optional
	 *            indicates if this data field is optional
	 * @return an instance of LastReq.
	 * 
	 */

	public static LastReq parse(DerInputStream data, byte explicitTag, boolean optional, Config kerberosConfig) throws Asn1Exception, IOException {
		if ((optional) && (((byte) data.peekByte() & (byte) 0x1F) != explicitTag))
			return null;
		DerValue der = data.getDerValue();
		if (explicitTag != (der.getTag() & (byte) 0x1F)) {
			throw new Asn1Exception(Krb5.ASN1_BAD_ID);
		} else {
			DerValue subDer = der.getData().getDerValue();
			return new LastReq(subDer, kerberosConfig);
		}
	}

}
