/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.scaffolder.model;

import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConstructElementDeclaration;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.mule.soapkit.scaffolder.declaration.builders.HttpArtifactDeclarationBuilder.LISTENER_CONFIGURATION_TAG_NAME;
import static org.mule.soapkit.scaffolder.declaration.builders.MuleArtifactDeclarationBuilder.FLOW_TAG_NAME;
import static org.mule.soapkit.scaffolder.declaration.builders.SoapkitArtifactDeclarationBuilder.CONFIGURATION_TAG_NAME;

public class MuleConfig implements MuleElement<ArtifactDeclaration> {

  private final ArtifactDeclaration declaration;

  public MuleConfig(ArtifactDeclaration declaration) {
    this.declaration = declaration;
  }

  public List<SoapkitConfig> soapkitConfigs() {
    return listSoapkitConfigs().stream().map(SoapkitConfig::new).collect(Collectors.toList());
  }

  public Optional<SoapkitConfig> apiConfig(String name) {
    for (SoapkitConfig config : soapkitConfigs()) {
      if (config.getName().equals(name))
        return Optional.of(config);
    }
    return Optional.empty();
  }

  public Optional<Flow> getSoapkitFlow(String configName) {
    return listFlows().stream()
        .filter(d -> d.getComponents().stream()
            .anyMatch(c -> "APIKit for SOAP".equals(c.getDeclaringExtension()) && "router".equals(c.getName())
                && configName.equals(c.getConfigRef())))
        .map(Flow::new)
        .findFirst();
  }

  public List<String> flowNames() {
    return listFlows().stream().map(ConstructElementDeclaration::getRefName).collect(Collectors.toList());
  }

  public List<HttpListenerConfig> httpListenerConfigs() {
    final List<ConfigurationElementDeclaration> httpListenerConfigs = listHttpListenerConfigs();
    return httpListenerConfigs.stream().map(HttpListenerConfig::new).collect(Collectors.toList());
  }

  @Override
  public ArtifactDeclaration getDeclaration() {
    return declaration;
  }

  @Override
  public String getName() {
    return declaration.getName();
  }

  private List<ConfigurationElementDeclaration> listHttpListenerConfigs() {
    return declaration.getGlobalElements().stream()
        .filter(ged -> LISTENER_CONFIGURATION_TAG_NAME.equals(ged.getName()))
        .map(ged -> (ConfigurationElementDeclaration) ged)
        .collect(Collectors.toList());
  }

  private List<ConstructElementDeclaration> listFlows() {
    return declaration.getGlobalElements().stream()
        .filter(ged -> FLOW_TAG_NAME.equals(ged.getName()))
        .map(ged -> (ConstructElementDeclaration) ged)
        .collect(Collectors.toList());
  }

  private List<ConfigurationElementDeclaration> listSoapkitConfigs() {
    return declaration.getGlobalElements().stream()
        .filter(ged -> CONFIGURATION_TAG_NAME.equals(ged.getName()))
        .map(ged -> (ConfigurationElementDeclaration) ged)
        .collect(Collectors.toList());
  }
}
