/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.scaffolder;

import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.soapkit.scaffolder.model.Flow;
import org.mule.soapkit.scaffolder.model.HttpListenerConfig;
import org.mule.soapkit.scaffolder.model.SoapkitConfig;
import org.mule.soapkit.scaffolder.model.SoapkitFlow;
import org.mule.soapkit.scaffolder.model.builders.MuleConfigBuilder;
import org.mule.soapkit.scaffolder.model.MuleDomain;
import org.mule.soapkit.scaffolder.model.MuleConfig;
import org.mule.soapkit.scaffolder.parsers.WsdlParser;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

public class Scaffolder {

  private Scaffolder() {}

  /**
   * @return an instance of the SOAP scaffolder.
   */
  public static Scaffolder getInstance() {
    return new Scaffolder();
  }

  /**
   * Scaffolds a MuleConfigBuilder given an existing one. Adds flows for the missing operations.
   * @param baseMuleConfig the existing Mule config.
   * @param wsdlLocation the absolute path of the wsdl file.
   * @param config the soapkit config, contains the service and port attributes to find the operations.
   * @return The scaffolded MuleConfigBuilder as an XML {@link String}
   */
  public ArtifactDeclaration scaffold(ArtifactDeclaration baseMuleConfig, String wsdlLocation, SoapkitConfig config) {
    return scaffold(baseMuleConfig, wsdlLocation, config, null);
  }

  /**
   * Scaffolds a MuleConfigBuilder given an existing one. Adds flows for the missing operations.
   * @param baseMuleConfig the existing Mule config.
   * @param wsdlLocation the absolute path of the wsdl file.
   * @param config the soapkit config, contains the service and port attributes to find the operations.
   * @param domainConfig the domain config declaration.
   * @return The scaffolded MuleConfigBuilder as an XML {@link String}
   */
  public ArtifactDeclaration scaffold(ArtifactDeclaration baseMuleConfig, String wsdlLocation, SoapkitConfig config,
                                      ArtifactDeclaration domainConfig) {
    final MuleConfig muleConfig = parseMuleConfig(baseMuleConfig);
    final WsdlParser apiParser = WsdlParser.create(wsdlLocation, config);

    final Optional<SoapkitConfig> apiConfigs =
        !muleConfig.soapkitConfigs().contains(config) ? Optional.ofNullable(config) : Optional.empty();

    final List<String> flows = muleConfig.flowNames();
    final List<Flow> newFlows = apiParser.flows().stream().filter(f -> !flows.contains(f.getName())).collect(Collectors.toList());

    List<HttpListenerConfig> newHttpListenerConfigs = emptyList();
    if (routerIsMissingForConfig(muleConfig, config.getName())) {
      final MuleDomain domainParser = new MuleDomain(domainConfig);
      final List<HttpListenerConfig> domainListenerConfigs = domainParser.httpListenerConfigs();
      final List<HttpListenerConfig> projectListenerConfigs = muleConfig.httpListenerConfigs();
      if (domainListenerConfigs.isEmpty() && projectListenerConfigs.isEmpty()) {
        newHttpListenerConfigs = singletonList(new HttpListenerConfig());
        newFlows.add(0, new SoapkitFlow(apiParser.config(), newHttpListenerConfigs));
      } else {
        newHttpListenerConfigs = emptyList();
        newFlows.add(0, new SoapkitFlow(apiParser.config(),
                                        projectListenerConfigs.isEmpty() ? domainListenerConfigs : projectListenerConfigs));
      }
    }

    return new MuleConfigBuilder(muleConfig.getDeclaration())
        .withConfig(apiConfigs.orElse(null))
        .withFlows(newFlows)
        .withHttpListeners(newHttpListenerConfigs)
        .build();
  }

  private boolean routerIsMissingForConfig(MuleConfig muleConfig, String configName) {
    return !muleConfig.getSoapkitFlow(configName).isPresent();
  }

  /**
   * Scaffolds a new MuleConfigBuilder. The port and service defines a binding with the operations to scaffold
   * @param wsdlPath the absolute path of the wsdl file.
   * @param service the selected wsdl service.
   * @param port the selected service port.
   * @param domainConfig the domain config declaration.
   * @return The scaffolded MuleConfigBuilder as an XML {@link String}
   */
  public ArtifactDeclaration scaffold(String wsdlPath, String wsdlLocationAttribute, String service, String port,
                                      ArtifactDeclaration domainConfig) {
    final WsdlParser apiParser = WsdlParser.create(wsdlPath, wsdlLocationAttribute, service, port);
    final MuleDomain domainParser = new MuleDomain(domainConfig);

    final SoapkitConfig config = apiParser.config();

    final SoapkitFlow soapkitFlow;
    final List<HttpListenerConfig> httpListenerConfigs;
    if (domainParser.httpListenerConfigs().isEmpty()) {
      httpListenerConfigs = singletonList(new HttpListenerConfig());
      soapkitFlow = new SoapkitFlow(apiParser.config(), httpListenerConfigs);
    } else {
      httpListenerConfigs = emptyList();
      soapkitFlow = new SoapkitFlow(apiParser.config(), domainParser.httpListenerConfigs());
    }

    final List<Flow> flows = new ArrayList<>(singletonList(soapkitFlow));
    flows.addAll(apiParser.flows());

    return new MuleConfigBuilder()
        .withConfig(config)
        .withFlows(flows)
        .withHttpListeners(httpListenerConfigs)
        .build();
  }

  /**
   * Parses a MuleConfigBuilder xml file
   * @param muleConfig The mule config declaration
   * @return A representation of the Mule Config
   */
  public MuleConfig parseMuleConfig(ArtifactDeclaration muleConfig) {
    return new MuleConfig(muleConfig);
  }

}
