/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.scaffolder.model;

import org.apache.commons.lang3.math.NumberUtils;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;

import static org.mule.soapkit.scaffolder.declaration.builders.HttpArtifactDeclarationBuilder.PARAMETER_HOST;
import static org.mule.soapkit.scaffolder.declaration.builders.HttpArtifactDeclarationBuilder.PARAMETER_PORT;
import static org.mule.soapkit.scaffolder.declaration.builders.HttpArtifactDeclarationBuilder.buildHttpListenerConfig;
import static org.mule.soapkit.scaffolder.declaration.builders.HttpArtifactDeclarationBuilder.buildHttpListenerConnection;
import static org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel.CONNECTION;

public class HttpListenerConfig implements MuleElement<ConfigurationElementDeclaration> {

  static final String DEFAULT_PORT = "8081";
  static final String DEFAULT_HOST = "0.0.0.0";

  private HttpListenerConnection httpConnection;
  private ConfigurationElementDeclaration declaration;

  public HttpListenerConfig() {
    this((String) null);
  }

  public HttpListenerConfig(String basePath) {
    this(buildHttpListenerConfig(basePath, new HttpListenerConnection().declaration));
  }

  public HttpListenerConfig(ConfigurationElementDeclaration declaration) {
    this.declaration = declaration;
  }

  @Override
  public String getName() {
    return declaration.getRefName();
  }

  @Override
  public ConfigurationElementDeclaration getDeclaration() {
    return declaration;
  }

  private HttpListenerConnection getHttpConnection() {
    if (httpConnection == null) {
      httpConnection = declaration.getConnection()
          .map(HttpListenerConnection::new)
          .orElse(new HttpListenerConnection());
    }

    return httpConnection;
  }

  public String getHost() {
    return getHttpConnection().getHost();
  }

  public int getPort() {
    return NumberUtils.toInt(getHttpConnection().getPort(), 0);
  }

  private static class HttpListenerConnection {

    private ConnectionElementDeclaration declaration;

    HttpListenerConnection() {
      this(DEFAULT_HOST, DEFAULT_PORT);
    }

    HttpListenerConnection(String host, String port) {
      this(buildHttpListenerConnection(host, port));
    }

    HttpListenerConnection(ConnectionElementDeclaration declaration) {
      this.declaration = declaration;
    }

    public String getHost() {
      return getParameterValue(PARAMETER_HOST);
    }

    public String getPort() {
      return getParameterValue(PARAMETER_PORT);
    }

    private String getParameterValue(String parameter) {
      return declaration.getParameterGroup(CONNECTION)
          .flatMap(pg -> pg.getParameter(parameter))
          .map(pd -> pd.getValue().toString())
          .orElse(null);
    }
  }

}
