/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.scaffolder.parsers;

import org.apache.commons.lang3.StringUtils;
import org.mule.soapkit.scaffolder.model.Flow;
import org.mule.soapkit.scaffolder.model.SoapkitConfig;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.mule.wsdl.parser.model.operation.OperationModel;

import java.util.Collections;
import java.util.List;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;

public class WsdlParser {

  private final SoapkitConfig config;
  private final WsdlModel wsdlModel;

  private WsdlParser(final String wsdlLocation, final SoapkitConfig config) {
    wsdlModel = parseWSDL(wsdlLocation);
    validateConfig(config);
    this.config = config;
  }

  public static WsdlParser create(final String wsdlLocation, final SoapkitConfig config) {
    return new WsdlParser(wsdlLocation, config);
  }

  private void validateConfig(final SoapkitConfig config) {
    if (config.getWsdlPath() == null)
      throw new IllegalArgumentException("The given 'wsdLocation' for the SimpleServiceDefinition must not be null (the provided wsdl file is probably unreachable)");
    if (StringUtils.isBlank(config.getService()))
      throw new IllegalArgumentException("The 'service' attribute cannot be blank");
    if (StringUtils.isBlank(config.getPort()))
      throw new IllegalArgumentException("The 'port' attribute cannot be blank");
  }

  public static WsdlParser create(final String wsdlLocation, final String wsdlPath, final String serviceName,
                                  final String portName) {
    return new WsdlParser(wsdlLocation, new SoapkitConfig(wsdlPath, serviceName, portName));
  }

  public List<String> operationNames() {
    final ServiceModel service = wsdlModel.getService(config.getService());
    if (service == null)
      return Collections.emptyList();

    final PortModel port = service.getPort(config.getPort());
    if (port == null)
      return Collections.emptyList();

    return port.getOperations().stream().map(OperationModel::getName).collect(toList());
  }

  public List<String> flowNames() {
    return operationNames().stream().map(name -> name + ":\\" + config.getName()).collect(toList());
  }

  public List<Flow> flows() {
    return flowNames().stream().map(Flow::new).collect(toList());
  }

  public SoapkitConfig config() {
    return config;
  }


  private static WsdlModel parseWSDL(final String wsdlLocation) {
    try {
      validateBlankString(wsdlLocation, "wsdl location");
      return org.mule.wsdl.parser.WsdlParser.Companion.parse(wsdlLocation);
    } catch (final Exception e) {
      final String msg = String.format("Something went wrong when parsing the wsdl file %s",
                                       isBlank(e.getMessage()) ? " for [" + wsdlLocation + "]"
                                           : ", full message " + e.getMessage());
      throw new IllegalArgumentException(msg, e);
    }
  }

  private static void validateBlankString(final String paramValue, final String paramName) {
    if (isBlank(paramValue)) {
      throw new IllegalArgumentException("The " + paramName + " can not be blank nor null.");
    }
  }
}
