/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tools.muleesb;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import groovy.lang.GroovyShell;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.AbstractArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

public abstract class AbstractMuleMojo extends AbstractMojo
{

    @Component
    protected MavenProject mavenProject;

    @Component
    protected ArtifactFactory artifactFactory;

    @Component
    protected ArtifactResolver artifactResolver;

    @Parameter(defaultValue = "${localRepository}", readonly = true)
    protected ArtifactRepository localRepository;

    @Parameter (property = "mule.skip")
    protected String skip;

    @Parameter
    protected File domain;

    @Parameter(property = "script", required = false)
    protected File script;

    @Parameter(property = "mule.timeout", required = false)
    protected int timeout;

    @Parameter
    private List<ArtifactDescription> artifactItems = new ArrayList<ArtifactDescription>();

    /**
     * @see org.apache.maven.plugin.Mojo#execute()
     */
    @Override
    public void execute() throws MojoExecutionException, MojoFailureException
    {
        if (StringUtils.isNotEmpty(skip) && "true".equals(skip)) {
            getLog().info("Skipping execution: skip=" + skip);
        } else {
            getLog().debug("Executing mojo, skip=" + skip);
            doExecute();
        }
    }

    protected abstract void doExecute() throws MojoFailureException, MojoExecutionException;

    protected File getDependency(ArtifactDescription artifactDescription)
            throws MojoExecutionException, MojoFailureException
    {
        try
        {
            Artifact artifact = artifactFactory.createArtifact(artifactDescription.getGroupId(),
                                                               artifactDescription.getArtifactId(), artifactDescription.getVersion(), null,
                                                               artifactDescription.getType());
            getLog().info("Resolving " + artifact);
            artifactResolver.resolve(artifact, mavenProject.getRemoteArtifactRepositories(), localRepository);
            return artifact.getFile();
        }
        catch (AbstractArtifactResolutionException e)
        {
            throw new MojoExecutionException("Couldn't download artifact: " + e.getMessage(), e);
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Couldn't download artifact: " + e.getMessage());
        }
    }


    protected void addDomain(Deployer deployer) throws MojoFailureException
    {
        if (domain != null && domain.exists()) {
            getLog().debug("Adding domain with configuration: " + domain);
            deployer.addDomain(domain);
        }
        else
        {
            getLog().debug("Domain configuration not found: " + domain);
        }
    }

    protected String readFile( String file ) throws IOException
    {
        BufferedReader reader = new BufferedReader( new FileReader(file));
        String line;
        StringBuilder  stringBuilder = new StringBuilder();

        while( ( line = reader.readLine() ) != null ) {
            stringBuilder.append( line );
            stringBuilder.append( "\n" );
        }

        return stringBuilder.toString();
    }

    protected void executeGroovyScript() throws MojoExecutionException
    {
        GroovyShell shell = new GroovyShell();
        getLog().info("executing script: " + script.getAbsolutePath());
        shell.setProperty("basedir",mavenProject.getBasedir());

        getLog().info(mavenProject.getBasedir().getAbsolutePath());
        try
        {
            shell.evaluate(readFile(script.getAbsolutePath()));
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("error executing script: " + script.getAbsolutePath() + "\n" + e.getMessage() );
        }
    }

    protected void addDependencies(Deployer deployer) throws MojoFailureException, MojoExecutionException
    {
        List<File> libraries = new ArrayList<File>();
        for (ArtifactDescription artifact: artifactItems)
        {
            libraries.add(this.getDependency(artifact));
        }
        deployer.addLibraries(libraries);
    }
}
