/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tools.muleesb;

import org.mule.test.infrastructure.process.MuleProcessController;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.UnArchiver;
import org.codehaus.plexus.archiver.manager.ArchiverManager;
import org.codehaus.plexus.archiver.manager.NoSuchArchiverException;

/**
 * Installs and starts a Mule ESB cluster and deploys a list of applications.
 * Downloads a Mule ESB distribution Enterprise, from a configured Maven repository (Available in pom.xml or settings.xml). You can specify the full Maven coordinates using <code>muleDistribution</code> tag, or you can just configure <code>muleVersion</code> tag and let the plugin use the default groupId and artifactId.
 * Optionally you can specify a list of Mule applications to be deployed, if you don't configure it, it will try to deploy the primary packaged artifact for the Maven project.
 * You can send properties to start each Mule Server (don't forget to use -M-D for Mule properties).
 * You can configure a deployment timeout using <code>deploymentTimeout</code> tag.
 *
 * @author <a href="mailto:asequeira@gmail.com">Ale Sequeira</a>
 * @see org.mule.tools.muleesb.ClusterUndeployMojo
 * @see org.mule.test.infrastructure.process.MuleProcessController
 * @since 1.0
 */
@Mojo(name = "clusterDeploy", requiresProject = false, defaultPhase = LifecyclePhase.PRE_INTEGRATION_TEST)
public class ClusterDeployMojo extends AbstractMuleMojo
{

    private static final long DEFAULT_POLLING_DELAY = 1000;

    @Component
    private ArtifactRepositoryFactory artifactRepositoryFactory;

    @Component(role = ArtifactRepositoryLayout.class)
    private Map<String, ArtifactRepositoryLayout> repositoryLayouts;

    @Component
    private ArtifactMetadataSource source;

    @Component
    protected ArchiverManager archiverManager;

    /**
     * Number of cluster nodes.
     *
     * @since 1.0
     */
    @Parameter(defaultValue = "2", readonly = true, required = true)
    private Integer clusterSize;

    /**
     * Maven coordinates for the Mule ESB distribution to download.
     * You need to specify:
     * <li>groupId</li>
     * <li>artifactId</li>
     * <li>version</li>
     * This parameter and <code>muleVersion</code> are mutual exclusive
     *
     * @since 1.0
     */
    @Parameter(readonly = true)
    private ArtifactDescription muleDistribution;

    /**
     * Version of the Mule ESB Enterprise distribution to download. If you need to use Community version use <code>muleDistribution</code> parameter.
     * This parameter and <code>muleDistribution</code> are mutual exclusive.
     *
     * @since 1.0
     */
    @Parameter(readonly = true, property = "mule.version")
    private String muleVersion;

    /**
     * List of applications to be deployed to Mule ESB cluster. Each application is the full path to a zipped or exploded Mule application.
     *
     * @since 1.0
     */
    @Parameter(property = "mule.applications", required = true)
    private List<File> applications;

    /**
     * Deployment timeout in milliseconds.
     *
     * @since 1.0
     */
    @Parameter(property = "mule.deployment.timeout", defaultValue = "60000", required = true)
    private Long deploymentTimeout;

    /**
     * List of Mule ESB Standalone command line arguments.
     * Adding a property to this list is the same that adding it to the command line when starting Mule using bin/mule.
     * If you want to add a Mule property don't forget to prepend <code>-M-D</code>.
     * If you want to add a System property for the Wrapper don't forget to prepend <code>-D</code>.
     *
     * Example: <code><arguments><argument>-M-Djdbc.url=jdbc:oracle:thin:@myhost:1521:orcl</argument></arguments></code>
     *
     * @since 1.0
     */
    @Parameter(property = "mule.arguments", required = false)
    private String[] arguments;

    /**
     * List of external libs (Jar files) to be added to MULE_HOME/user/lib directory.
     *
     * @since 1.0
     */
    @Parameter
    private List<File> libs = new ArrayList<File>();

    public void doExecute() throws MojoExecutionException, MojoFailureException
    {
        if (muleDistribution == null) {
            muleDistribution = new ArtifactDescription("com.mulesoft.muleesb.distributions", "mule-ee-distribution-standalone", muleVersion, "tar.gz");
            this.getLog().info("muleDistribution not set, using default artifact: " + muleDistribution);
        }
        validate();
        File buildDirectory = new File(mavenProject.getBuild().getDirectory());
        installMules(muleDistribution, buildDirectory);

        List<MuleProcessController> controllers = new LinkedList<MuleProcessController>();
        File[] muleHomes = new File[clusterSize];

        for (int i = 0; i < clusterSize; i++)
        {
            muleHomes[i] = new File(mavenProject.getBuild().getDirectory() + "/mule" + i, "mule-enterprise-standalone-" + muleDistribution.getVersion());
            controllers.add(new MuleProcessController(muleHomes[i].getAbsolutePath(), timeout));
            validateEnterpriseVersion(muleHomes[i]);
        }
        ClusterConfigurator configurator = new ClusterConfigurator();
        if (null != script) executeGroovyScript();
        new ClusterDeployer(muleHomes, controllers, getLog(), applications, deploymentTimeout, arguments, DEFAULT_POLLING_DELAY,configurator).addLibraries(libs).execute();
    }

    private void validateEnterpriseVersion(File home) throws MojoFailureException
    {
        Scanner scanner = null;
        try
        {
            scanner = new Scanner(new File(home, "LICENSE.txt"));
        }
        catch (FileNotFoundException e)
        {
            throw new MojoFailureException("Cannot create cluster in Community Edition.\nCannot find license file in: " + new File(home, "LICENSE.txt").getAbsolutePath());
        }
        while (scanner.hasNextLine())
        {
            if (scanner.nextLine().contains("CPAL") )
            {
                throw new MojoFailureException("Cannot create cluster in Community Edition.");
            }
        }
    }


    private void validate() throws MojoFailureException
    {
        verify(muleDistribution != null, "Mule ESB artifact description does not exist.");
        verifyNotNull(muleDistribution.getGroupId(), "groupId");
        verifyNotNull(muleDistribution.getArtifactId(), "artifactId");
        verifyNotNull(muleDistribution.getVersion(), "version");
        verifyNotNull(muleDistribution.getType(), "type");
    }

    private void installMules(ArtifactDescription muleDistribution, File buildDirectory)
            throws MojoExecutionException, MojoFailureException
    {
        for (int i = 0; i < clusterSize; i++)
        {
            File destDir = new File(buildDirectory, "/mule" + i);
            destDir.mkdir();
            installMule(muleDistribution, destDir);
        }
    }

    /**
     * This code was inspired by maven-dependency-plugin GetMojo.
     */
    private void installMule(ArtifactDescription muleDistribution, File destDir)
            throws MojoExecutionException, MojoFailureException
    {
        File src = getDependency(muleDistribution);
        getLog().info("Copying " + src.getAbsolutePath() + " to " + destDir.getAbsolutePath());
        extract(src, destDir, muleDistribution.getType());
    }

    private void extract(File src, File dest, String type)
            throws MojoExecutionException, MojoFailureException
    {
        try
        {
            UnArchiver unArchiver = getArchiver(type);
            unArchiver.setSourceFile(src);
            unArchiver.setDestDirectory(dest);
            unArchiver.extract();
        }
        catch (ArchiverException e)
        {
            throw new MojoExecutionException("Couldn't extract file " + src + " to " + dest);
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Couldn't extract file " + src + " to " + dest);
        }
    }

    private UnArchiver getArchiver(String type) throws MojoExecutionException
    {
        UnArchiver unArchiver;
        try
        {
            unArchiver = archiverManager.getUnArchiver(type);
            getLog().debug("Found unArchiver by extension: " + unArchiver);
        }
        catch (NoSuchArchiverException e)
        {
            throw new MojoExecutionException("Couldn't find archiver for type: " + type);
        }
        return unArchiver;
    }

    private void verifyNotNull(String value, String name) throws MojoFailureException
    {
        verify(StringUtils.isNotEmpty(value), "muleDistribution." + name + " cannot be empty.");
    }

    private void verify(boolean exists, String message) throws MojoFailureException
    {
        if (!exists)
        {
            throw new MojoFailureException(message);
        }
    }
    
}
