/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tools.muleesb;

import org.mule.test.infrastructure.process.MuleProcessController;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.UnArchiver;
import org.codehaus.plexus.archiver.manager.ArchiverManager;
import org.codehaus.plexus.archiver.manager.NoSuchArchiverException;

/**
 * Installs and starts a Mule ESB Standalone server and deploys a list of applications.
 * Downloads a Mule ESB distribution, CE or EE, from a configured Maven repository (Available in pom.xml or settings.xml). You can specify the full Maven coordinates using <code>muleDistribution</code> tag, or you can just configure <code>muleVersion</code> tag and let the plugin use the default groupId and artifactId for the Enterprise distribution.
 * Optionally you can specify a list of Mule applications to be deployed, if you don't configure it, it will try to deploy the primary packaged artifact for the Maven project.
 * You can send properties to start the Mule Server (don't forget to use -M-D for Mule properties).
 * You can configure a deployment timeout using <code>deploymentTimeout</code> tag.
 *
 * @author <a href="mailto:asequeira@gmail.com">Ale Sequeira</a>
 * @see org.mule.tools.muleesb.UndeployMojo
 * @see org.mule.tools.muleesb.Undeployer
 * @see org.mule.test.infrastructure.process.MuleProcessController
 * @since 1.0
 */
@Mojo(name = "deploy", requiresProject = true, defaultPhase = LifecyclePhase.PRE_INTEGRATION_TEST)
public class DeployMojo extends AbstractMuleMojo    
{

    private static final long DEFAULT_POLLING_DELAY = 1000;

    @Component
    private ArtifactRepositoryFactory artifactRepositoryFactory;

    @Component(role = ArtifactRepositoryLayout.class)
    private Map<String, ArtifactRepositoryLayout> repositoryLayouts;

    @Component
    private ArtifactMetadataSource source;

    @Component
    protected ArchiverManager archiverManager;

    /**
     * Version of the Mule ESB Enterprise distribution to download. If you need to use Community version use <code>muleDistribution</code> parameter.
     * This parameter and <code>muleDistribution</code> are mutual exclusive.
     *
     * @since 1.0
     */
    @Parameter(readonly = true, property = "mule.version")
    private String muleVersion;


    /**
     * Maven coordinates for the Mule ESB distribution to download.
     * You need to specify:
     * <li>groupId</li>
     * <li>artifactId</li>
     * <li>version</li>
     * This parameter and <code>muleVersion</code> are mutual exclusive
     *
     * @since 1.0
     */
    @Parameter(readonly = true)
    private ArtifactDescription muleDistribution;

    /**
     * List of applications to be deployed to Mule ESB. Each application is the full path to a zipped or exploded Mule application.
     *
     * @since 1.0
     */
    @Parameter(property = "mule.applications", required = true)
    private List<File> applications;

    /**
     * Deployment timeout in milliseconds.
     *
     * @since 1.0
     */
    @Parameter(property = "mule.deployment.timeout", defaultValue = "60000", required = true)
    private Long deploymentTimeout;

    /**
     * List of Mule ESB Standalone command line arguments.
     * Adding a property to this list is the same that adding it to the command line when starting Mule using bin/mule.
     * If you want to add a Mule property don't forget to prepend <code>-M-D</code>.
     * If you want to add a System property for the Wrapper don't forget to prepend <code>-D</code>.
     *
     * Example: <code><arguments><argument>-M-Djdbc.url=jdbc:oracle:thin:@myhost:1521:orcl</argument></arguments></code>
     *
     * @since 1.0
     */
    @Parameter(property = "mule.arguments", required = false)
    private String[] arguments;

    /**
     * List of external libs (Jar files) to be added to MULE_HOME/user/lib directory.
     *
     * @since 1.0
     */
    @Parameter
    private List<File> libs = new ArrayList<File>();

    private HashMap<String, String> distributionMap;

    public void doExecute() throws MojoExecutionException, MojoFailureException
    {
        if (muleDistribution == null) {
            muleDistribution = new ArtifactDescription("com.mulesoft.muleesb.distributions", "mule-ee-distribution-standalone", muleVersion, "tar.gz");
            this.getLog().info("muleDistribution not set, using default artifact: " + muleDistribution);
        }
        setup();
        validate();
        File buildDirectory = new File(mavenProject.getBuild().getDirectory());
        installMule(muleDistribution, buildDirectory);
        File muleHome = new File(buildDirectory, distributionMap.get(muleDistribution.getArtifactId())
                                                 + muleDistribution.getVersion());
        mavenProject.getProperties().setProperty("mule.home", muleHome.getAbsolutePath());
        getLog().info("Using MULE_HOME: " + muleHome);
        MuleProcessController mule = new MuleProcessController(muleHome.getAbsolutePath(), timeout);
        if (applications.isEmpty()) {
            applications = new ArrayList<File>();
            Artifact artifact = artifactFactory.createArtifact(mavenProject.getGroupId(), mavenProject.getArtifactId(), mavenProject.getVersion(), "", "zip");
            try
            {
                artifactResolver.resolve(artifact, new ArrayList<ArtifactRepository>(), localRepository);
            }
            catch (ArtifactResolutionException e)
            {
                throw new MojoFailureException("Couldn't find Mule application.");
            }
            catch (ArtifactNotFoundException e)
            {
                throw new MojoFailureException("Couldn't find Mule application.");
            }
            applications.add(artifact.getFile());
            getLog().info("No application configured. Using project artifact: " + artifact.getFile());
        }
        Deployer deployer = new Deployer(mule, getLog(), applications, deploymentTimeout, arguments, DEFAULT_POLLING_DELAY)
                .addLibraries(libs);
        addDomain(deployer);
        addDependencies(deployer);
        if (null != script) executeGroovyScript();
        deployer.execute();
    }

    private void setup()
    {
        distributionMap = new HashMap<String,String>();
        distributionMap.put("mule-ee-distribution-standalone", "mule-enterprise-standalone-");
        distributionMap.put("mule-standalone", "mule-standalone-");
    }

    private void validate() throws MojoFailureException
    {
        verify(muleDistribution != null, "Mule ESB artifact description does not exist.");
        verifyNotNull(muleDistribution.getGroupId(), "groupId");
        verifyNotNull(muleDistribution.getArtifactId(), "artifactId");
        verifyNotNull(muleDistribution.getVersion(), "version");
        verifyNotNull(muleDistribution.getType(), "type");
    }

    /**
     * This code was inspired by maven-dependency-plugin GetMojo.
     */
    private void installMule(ArtifactDescription muleDistribution, File destDir)
        throws MojoExecutionException, MojoFailureException
    {
        File src = getDependency(muleDistribution);
        getLog().info("Copying " + src.getAbsolutePath() + " to " + destDir.getAbsolutePath());
        extract(src, destDir, muleDistribution.getType());
    }

    private void extract(File src, File dest, String type)
        throws MojoExecutionException, MojoFailureException
    {
        try
        {
            UnArchiver unArchiver = getArchiver(type);
            unArchiver.setSourceFile(src);
            unArchiver.setDestDirectory(dest);
            unArchiver.extract();
        }
        catch (ArchiverException e)
        {
            throw new MojoExecutionException("Couldn't extract file " + src + " to " + dest);
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Couldn't extract file " + src + " to " + dest);
        }
    }

    private UnArchiver getArchiver(String type) throws MojoExecutionException
    {
        UnArchiver unArchiver;
        try
        {
            unArchiver = archiverManager.getUnArchiver(type);
            getLog().debug("Found unArchiver by extension: " + unArchiver);
        }
        catch (NoSuchArchiverException e)
        {
            throw new MojoExecutionException("Couldn't find archiver for type: " + type);
        }
        return unArchiver;
    }

    private void verifyNotNull(String value, String name) throws MojoFailureException
    {
        verify(StringUtils.isNotEmpty(value), "muleDistribution." + name + " cannot be empty.");
    }

    private void verify(boolean exists, String message) throws MojoFailureException
    {
        if (!exists)
        {
            throw new MojoFailureException(message);
        }
    }





}
