package org.mule.tools.report;

import org.mule.tools.RAMLAnalyzer;
import org.mule.tools.report.model.Testsuite;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;


public class RAMLReportCreator {

    private static final String VALIDATION_FAILURE_TYPE = "Validation failure";
    private static final String OUTPUT_ANALYSIS_DIR = "raml-analysis-reports";
    private static final String OUTPUT_ANALYSIS_PREFIX = "REPORT.";

    public static void generateReports(Path filePath) throws JAXBException, IOException {
        RAMLAnalyzer analyzer = new RAMLAnalyzer(filePath.toFile());
        Analysis analysisReport = analyzer.analyze();

        Testsuite testsuite = new Testsuite();
        List<Testsuite.Testcase> testcase = testsuite.getTestcase();
        int successCount = 0;
        int failureCount = 0;

        for (RuleResult ruleResult : analysisReport.getResults()) {
            if(ruleResult.isSuccessful()) {
                testcase.add(getPassedTestsuite(ruleResult));
                successCount++;
            }
            else {
                testcase.add(getFailedTestsuite(ruleResult));
                failureCount++;
            }
        }

        fillSuiteInformation(testsuite, filePath, successCount, failureCount);
        Path outputFile = getOutputFile(filePath);
        createXML(testsuite, outputFile);
    }

    /* Method to fill information required by the Surefire schema. */
    private static void fillSuiteInformation(Testsuite testsuite, Path filePath, int successCount, int failureCount) {
        testsuite.setName(filePath.getFileName().toString());
        testsuite.setFailures(String.valueOf(failureCount));
        testsuite.setTests(String.valueOf(successCount + failureCount));
        testsuite.setSkipped("0");
        testsuite.setErrors("0");
    }

    private static Path getOutputFile(Path filePath) {
        Path outputDirectory = filePath.getParent().resolve(OUTPUT_ANALYSIS_DIR);
        String fileName = OUTPUT_ANALYSIS_PREFIX + filePath.getFileName().toString() + ".xml";
        return outputDirectory.resolve(fileName);
    }

    private static void createXML(Testsuite testsuite, Path outputFile) throws JAXBException, IOException {
        Files.createDirectories(outputFile.getParent());

        JAXBContext jc = JAXBContext.newInstance(Testsuite.class);
        Marshaller marshaller = jc.createMarshaller();
        marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

        FileOutputStream output = new FileOutputStream(outputFile.toFile());
        marshaller.marshal(testsuite, output);
    }

    private static Testsuite.Testcase getFailedTestsuite(RuleResult result) {
        Testsuite.Testcase testcase = new Testsuite.Testcase();
        testcase.setName(result.getName());
        testcase.setTime(getDurationString(result.getDuration()));

        Testsuite.Testcase.Failure failure = new Testsuite.Testcase.Failure();
        failure.setType(VALIDATION_FAILURE_TYPE);
        failure.setMessage(result.getFailure());
        testcase.getFailure().add(failure);

        return testcase;
    }

    private static Testsuite.Testcase getPassedTestsuite(RuleResult result) {
        Testsuite.Testcase testcase = new Testsuite.Testcase();
        testcase.setName(result.getName());
        testcase.setTime(getDurationString(result.getDuration()));
        return testcase;
    }

    private static String getDurationString(Duration duration) {
        // Returns the duration with the specification of the xml schema.
        // #seconds . #millis
        return String.format("%d.%03d", duration.getSeconds(), duration.getNano()/1000000);
    }

}
