/**
*
* This module contains the set of core Matchers to use in your tests
*
* .Example
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must beObject()
* ----
* Validates if a payload is of type Object
*
*/
%dw 2.0

import diff from dw::util::Diff
import some, every from dw::core::Arrays
import * from dw::test::internal::Utils



//TYPES
/**
* Data Type that represents the result of an Assertion
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* {
*   "matches": false,
*   description : { expected : "Number type", actual: "A Text" }
* }
* ----
*/
type MatcherResult = {
    "matches": Boolean,
    description: { expected: String, actual: String},
    reasons?: Array<String>
}

/**
* Data Type that represents a Matcher to perform assertions
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* %dw 2.0
* import dw::tests::Asserts
*
* fun beEqualToOne(): Matcher<Any> =
*     (actual) -> do {
*         {
*             matches: actual == 1,
*             description: { expected: "To be 1", actual: write(actual) as String }
*         }
*     }
* ----
*/
type Matcher<T> = (value: T) -> MatcherResult

//CONSTANTS
/**
* Constant that represents a successful match
*/
var MATCHED = {"matches": true, description: {expected: "", actual: ""}}



/**
* This function allows to assert a value with with a list of Matcher of Expressions
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must [
*     beObject(),
*     $.foo is Null
* ]
* ----
*/
fun must<T>(value: T, matchExpressions: Array<(value:T) -> Matcher<T> | MatcherResult | Boolean>): MatcherResult = do {
    matchExpressions reduce (assertion, result = MATCHED) ->
        if(result.matches) do {
           var assertionResult = assertion(value) match {
               case is MatcherResult -> $
               case is Matcher -> $(value)
               case is Boolean ->  { matches: $, description: { expected: "`" ++ dw::Runtime::locationString(assertion) ++ "`", actual: toReadableText(value) as String} }
           }
           ---
           assertionResult
        } else
           result

}

/**
* This function allows to assert a value with a Matcher of Expressions
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must beObject()
* ----
*/
fun must<T>(value: T, matcher: (value: T) -> Matcher<T> | Boolean): MatcherResult = do {
    matcher(value) match {
        case is Matcher -> $(value)
        case is Boolean ->
            { matches: $, description: { expected: "`" ++ dw::Runtime::locationString(matcher) ++ "`", actual: toReadableText(value) as String} }
    }
}

/**
* Validates that the value doesn't satisfy the given matcher
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must notBe(equalTo(2))
* ----
*/
fun notBe<T>(matcher:Matcher<T>): Matcher<T> = (value) -> do {
    var matcherResult = matcher(value)
    ---
     {
        matches: not matcherResult.matches,
        description: {
            expected: "Not to be $(matcherResult.description.expected)",
            actual: matcherResult.description.actual
        }
     }
}

/**
* Validates that the value satisfies at least one of the given matchers
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must anyOf(beObject(), beString())
* ----
*/
fun anyOf(matchers: Array<Matcher<Any>>): Matcher<Any> = (value) -> do {
    var matchersResult = matchers map $(value)
    var matchesResult = matchersResult some $.matches
    ---
    {
      matches: matchesResult,
      description: {
        expected: "Any of " ++ (matchersResult map $.description.expected joinBy " or ") ,
        actual: "$(toReadableText(value) as String)"
      }
    }
}

/**
* Validates that a value is equal to another one
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* (1 + 2) must equalTo(3)
* ----
*/
fun equalTo(expected: Any, equalToConfig: {unordered?: Boolean} = {}): Matcher<Any> =
    (actual) -> do {
        var diffResult = diff(actual, expected, equalToConfig)
        ---
        {
          matches: diffResult.matches,
          description:
            {
               expected: "$(toReadableText(expected)) to be equal to",
               actual: "$(toReadableText(actual))"
            },
          reasons: diffResult.diffs map ((diff,index) -> "Expecting `$(diff.path)` to be $(diff.expected) but was $(diff.actual)")
        }
    }

/**
* Validates that the array has the given size
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, 4, 7] must haveSize(3)
* ----
*/
fun haveSize(expectedSize:Number): Matcher<Array | String | Object> =
    (actual) -> do {
        var actualSize = sizeOf(actual)
        ---
        {
            matches: actualSize == expectedSize,
            description: { expected: "size $(expectedSize)", actual: "$(actualSize)" }
        }
    }

/**
* Validates that a given value is of type Array
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, 4, 7] must beArray()
* ----
*/
fun beArray(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Array
        ---
        {
            matches: isArray,
            description: {expected: "Array type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value is of type String
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must beString()
* ----
*/
fun beString(): Matcher =
    (actual: Any) -> do {
        var isString = actual is String
        ---
        {
            matches: isString,
            description: {expected: "String type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value is of type Number
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 123 must beNumber()
* ----
*/
fun beNumber(): Matcher =
    (actual: Any) -> do {
        var isNumber = actual is Number
        ---
        {
            matches: isNumber,
            description: {expected: "Number type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value is of type Boolean
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* true must beBoolean()
* ----
*/
fun beBoolean(): Matcher =
    (actual: Any) -> do {
        var isBoolean = actual is Boolean
        ---
        {
            matches: isBoolean,
            description: {expected: "Boolean type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value is of type Object
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name : "Lionel", lastName: "Messi"} must beObject()
* ----
*/
fun beObject(): Matcher =
    (actual: Any) -> do {
        var isObject = actual is Object
        ---
        {
            matches: isObject,
            description: { expected: "Object type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value is of type Null
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* null must beNull()
* ----
*/
fun beNull(): Matcher =
    (actual: Any) -> do {
        var isNull = actual is Null
        ---
        {
            matches: isNull,
            description: { expected: "Null type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that a given value isn't of type Null
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must notBeNull()
* ----
*/
fun notBeNull(): Matcher =
    (actual: Any) -> do {
        var notNull = not (actual is Null)
        ---
        {
            matches: notNull,
            description: { expected: "Not Null type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validates that the asserted String contains the given String
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must contain("ex")
* ----
*/
fun contain(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual contains expected
        ---
        {
            matches: matchContains,
            description: {expected: " to contain $(expected)", actual: actual}
        }
    }

/**
* Validates that the asserted Array contains the given value
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, "A Text", true] must contain(1)
* ----
*/
fun contain(expected:Any): Matcher<Array<Any>> =
    (actual:Array<Any>) -> do {
        var matchContains = actual contains expected
        ---
        {
            matches: matchContains,
            description: {expected: " to contain $(expected)", actual: toReadableText(actual)}
        }
    }


/**
* Validates that the value is contained in the given Array
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must beOneOf([1, "A Text", true])
* ----
*/
fun beOneOf(expected:Array<Any>): Matcher =
    (actual: Any) -> do{
        var exists = expected contains actual
        ---
        {
            matches: exists,
            description:
                {
                    expected: "one of $(toReadableText(expected))",
                    actual: "$(toReadableText(actual))"
                },
        }
    }

/**
* Validates that the asserted String ends with the given String
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must endWith("xt")
* ----
*/
fun endWith(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual endsWith expected
        ---
        {
            matches: matchContains,
            description: {expected: " to end with $(expected)", actual: actual}
        }
    }

/**
* Validates that the asserted String starts with the given String
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must startWith("A")
* ----
*/
fun startWith(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual startsWith expected
        ---
        {
            matches: matchContains,
            description: {expected: " to start with $(expected)", actual: actual}
        }
    }

/**
* Validates that the value (String, Object or Array) is empty
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [] must beEmpty()
* ----
*/
fun beEmpty(): Matcher<String | Object | Array | Null> =
     (actual:String | Object | Array | Null) -> do {
         var matchContains =  actual is Null or isEmpty(actual default "")
         ---
         {
             matches: matchContains,
             description: {expected: " to be empty.", actual: toReadableText(actual) }
         }
     }

/**
* Validates that the String value is blank
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "  " must beBlank()
* ----
*/
fun beBlank(): Matcher<String | Null> =
     (actual: String | Null) -> do {
         var matchContains = actual is Null or isBlank(actual default "")
         ---
         {
             matches: matchContains,
             description: {expected: " to be blank", actual: actual default "null"}
         }
     }
/**
* Validates that the Object has the given key
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must haveKey("name")
* ----
*/
fun haveKey(keyName: String): Matcher<Object> =
    (actual: Object) -> do {
        var matchKeyPresent = actual[keyName]?
        ---
        {
            matches: matchKeyPresent,
            description: {expected: " to contain key $(keyName)", actual: toReadableText(actual) }
        }
    }

/**
* Validates that the Object has the given value
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must haveValue("Messi")
* ----
*/
fun haveValue(value: Any): Matcher<Object> =
    (actual: Object) -> do {
        var matchTheValue = actual pluck $ contains value
        ---
        {
            matches: matchTheValue,
            description: {expected: " to contain value $(write(value))", actual: toReadableText(actual) }
        }
    }

/**
* Validates that the given value is equal to the content of a resource file
*
* The resource file must belong to the classpath
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must equalToResource("user.json", "application/json")
* ----
*/
fun equalToResource(resourceName: String, contentType: String = "application/dw", readerProperties: Object = {}): Matcher<Any> =
    equalTo(readUrl("classpath://$(resourceName)", contentType, readerProperties))


/**
* Validates that the asserted Comparable value is greater than the given one
*
* Can be equal to when using the _inclusive_ argument
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 3 must beGreaterThan(2)
* ----
*/
fun beGreaterThan(expected: Comparable, inclusive: Boolean = false): Matcher<Comparable> =
    (actual: Comparable) -> do {
        var matchGreaterThan = if (inclusive) actual >= expected else actual > expected
        ---
        {
            matches: matchGreaterThan,
            description: { expected: " to be greater than $(expected)", actual: toReadableText(actual) }
        }
    }

/**
* Validates that the asserted Comparable value is lower than the given one
*
* Can be equal to when using the _inclusive_ argument
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must beLowerThan(2)
* ----
*/
fun beLowerThan(expected: Comparable, inclusive: Boolean = false): Matcher<Comparable> =
    (actual: Comparable) -> do {
        var matchGreaterThan = if (inclusive) actual <= expected else actual < expected
        ---
        {
            matches: matchGreaterThan,
            description: { expected: " to be lower than $(expected)", actual: toReadableText(actual) }
        }
    }


/**
* Validates that each item of the array satisfies the given matcher
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1,2,3] must eachItem(beNumber())
* ----
*/
fun eachItem(matcher: Matcher<Any>): Matcher<Array<Any>> =
    (actual: Array<Any>) -> do {

        type IndexMatchPair = {index: Number, result: MatcherResult}

        fun collectAllFailed<T>(list: Array<T>, accumulator: Array<IndexMatchPair> = [], index: Number = 0): Array<IndexMatchPair> = do {
            list match {
              case [] -> accumulator
              case [head ~ tail] -> do {
                var matcherResult = matcher(head)
                ---
                if(matcherResult.matches)
                  collectAllFailed(tail, accumulator, index + 1)
                else
                  collectAllFailed(tail, accumulator + {index: index, result : matcherResult }, index + 1)
              }
            }
        }

        var allFailed = collectAllFailed(actual)
        ---
        {
            matches: isEmpty(allFailed),
            description: {
                expected: "All items of the array to match",
                actual: "$(toReadableText(allFailed.index default [])) didn't match."
            },
            reasons: allFailed map (result) -> "Item $(result.index) $(result.result.description.expected) but was $(result.result.description.actual)"
        }
    }

/**
* Validates that at least one item of the array satisfies the given matcher
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, true, "a text"] must haveItem(beNumber())
* ----
*/
fun haveItem(matcher: Matcher<Any>): Matcher<Array<Any>> =
    (actual: Array<Any>) -> do {
        {
            matches: actual some matcher($).matches,
            description: {
                expected: "Any item of the array to match",
                actual: "None did match."
            }
        }
    }



