/**
*
* This module contains the set of core Matchers to use in your tests
*
* === Example
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must beObject()
* ----
* Validates if a payload is of type Object
*
*/
%dw 2.0

import location, locationString, Location, try, orElse from dw::Runtime
import some from dw::core::Arrays
import * from dw::test::internal::Utils

//TYPES
/**
* Data Type that represents the result of an Assertion
*
* === Example
*
* ==== Source
*
* [source, DataWeave,linenums]
* ----
* {
*     "matches": false,
*     description : 'Expected value to be of type Number but was "A Text"'
* }
* ----
*/
type MatcherResult = {
    "matches": Boolean,
    description: String,
    reasons?: Array<String>,
}

/**
* Data Type that represents a Matcher to perform assertions
*
* === Example
*
* ==== Source
*
* [source, DataWeave,linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
*
* fun beEqualToOne(): Matcher<Any> =
*     (actual) -> do {
*         {
*             matches: actual == 1,
*             description: "Expected value to be one but was $(write(actual)) as String"
*         }
*     }
* ----
*/
type Matcher<T> = (value: T) -> MatcherResult

//CONSTANTS
/**
* Constant that represents a successful match
*/
var MATCHED = { "matches": true, description: "" }

/**
* This function allows to assert a value with with a list of Matcher or Expressions
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value |
* | matchExpressions |
* |===
*
* === Example
*
* This example shows how to assert that a `payload` is of type `Object` and has a property `foo` that is `null`
*
* ==== Source
*
* [source, DataWeave,linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must [
*     beObject(),
*     $.foo is Null
* ]
* ----
**/
fun must<T>(value: T, matchExpressions: Array<(value:T) -> Matcher<T> | MatcherResult | Boolean>): MatcherResult = do {
    matchExpressions reduce (assertion, result = MATCHED) ->
        if (result.matches) do {
             var assertionResult = assertion(value) match {
                 case is MatcherResult -> $
                 case is Matcher -> $(value)
                 case is Boolean ->  {
                    matches: $,
                    description: "`$(dw::Runtime::locationString(assertion))` was $(toReadableText(value) as String)."
                }
             }
             ---
             assertionResult
        } else result
}

/**
* This function allows to assert a value with a Matcher of Expressions
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to be assert
* | matcher | The matcher to be used
* |===
*
* === Example
*
* This example shows how to assert that a `payload` is of type `Object`.
*
* ==== Source
*
* [source, DataWeave,linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* payload must beObject()
* ----
**/
fun must<T>(value: T, matcher: (value: T) -> Matcher<T> | Boolean): MatcherResult = do {
    var loc = location(matcher)
    var fileName = try(() -> (loc.sourceIdentifier splitBy "::")[-1] as String ++ ".dwl") orElse "Unknown"
    ---
    matcher(value) match {
        case is Matcher -> $(value) update {
          case reasons at .reasons -> reasons map (r) -> r ++ " From `$(locationString(matcher))` at line $(loc.start.line) of $fileName."
          case descr at .description -> descr ++ " From `$(locationString(matcher))` at line $(loc.start.line) of $fileName."
        }
        case is Boolean -> {
            "matches": $,
            description: "Assertion `$(dw::Runtime::locationString(matcher))` failed with value $(toReadableText(value) as String). From `$(locationString(matcher))` at line $(loc.start.line) of $fileName."
        }
    }
}

/**
*
* Validates that the value doesn't satisfy the given matcher
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | matcher | The matcher that is going to be negated
* |===
*
* === Example
*
* This example shows how to assert that a value must `not` be 2.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must notBe(equalTo(2))
* ----
**/
fun notBe<T>(matcher: Matcher<T>): Matcher<T> = (value) -> do {
    var matcherResult = matcher(value)
    ---
    {
        matches: not matcherResult.matches,
        description: "Expected not `$(matcherResult.description).`"
    }
}

/**
* Validates that the value satisfies at least one of the given matchers
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | matchers | The list of matchers to be tested.
* |===
*
* === Example
*
* This example shows how to assert that a value must be either an `Object` or a `String`.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must anyOf(beObject(), beString())
* ----
**/
fun anyOf(matchers: Array<Matcher<Any>>): Matcher<Any> = (value) -> do {
    var matchersResult = matchers map $(value)
    var matchesResult = matchersResult some $.matches
    ---
    {
        matches: matchesResult,
        description: 'Expected any of $(matchersResult map $.description joinBy " or ") but was $(toReadableText(value) as String).'
    }
}

/**
* Validates that a value is equal to another one
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The expected value
* | equalToConfig | Configuration of how to compare them.
* |===
*
* === Example
*
* This example shows how to assert that a value must be `equal` to `3`.
*
* ==== Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* (1 + 2) must equalTo(3)
* ----
**/
fun equalTo(expected: Any, equalToConfig: {unordered?: Boolean} = {}): Matcher<Any> = (actual) -> do {
    var diffResult = diff(actual, expected, equalToConfig)

    fun diffToString(diff: Difference): String = diff  match {
    	case is AttributeNotPresent -> "Expected attribute with type $($.valueType) at $($.path) but was not present."
    	case is UnexpectedAttribute -> "Encountered unexpected attribute with type $($.valueType) at $($.path)."
    	case is KeyNotPresent -> "Expected key with value type $($.valueType) at $($.path) but was not present."
    	case is UnexpectedKey -> "Encountered unexpected entry with type $($.valueType) at $($.path)."
    	case is TypeMismatch -> "Expected value to have type $($.expected) but was $($.actual) at $($.path)."
    	case is ValueMismatch -> "Expected value to be $(toReadableText($.expected)) but was $(toReadableText($.actual)) at $($.path)."
    	case is ArrayLengthMismatch -> "Expected size of array to be $($.expected) but was $($.actual) at $($.path)."
    }

    ---
    {
        matches: diffResult.matches,
        description: "Expected $(toReadableText(expected)) but was $(toReadableText(actual)).",
        reasons: diffResult.diffs map diffToString($),
    }
}

/**
* Validates that the array has the given size
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expectedSize | The expected array size
* |===
*
* === Example
*
* Expects that the array must be of size `3`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, 4, 7] must haveSize(3)
* ----
*
**/
fun haveSize(expectedSize: Number): Matcher<Array | String | Object | Null> = (actual) -> actual match {
    case is Null -> {
        matches: false,
        description: "Expected size $(expectedSize) but was null."
    }
    case is String | Array | Object -> do {
        var actualSize = sizeOf($)
        ---
        {
            matches: actualSize == expectedSize,
            description: "Expected size $(expectedSize) but was $(actualSize)."
        }
    }
}

/**
*  Validates that a given value is of type Array
*
* === Example
*
* This example shows how to validate that a value is of type 'Array'
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, 4, 7] must beArray()
* ----
**/
fun beArray(): Matcher = (actual: Any) ->
    {
        matches: actual is Array,
        description: "Expected value to be Array type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value is of type String
*
* === Example
*
* This example shows how to validate that a value is of type `String`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must beString()
* ----
*/
fun beString(): Matcher = (actual: Any) ->
    {
        matches: actual is String,
        description: "Expected value to be String type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value is of type Number
*
* === Example
*
* This example shows how to validate that a value is of type `Number`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 123 must beNumber()
* ----
*/
fun beNumber(): Matcher = (actual: Any) ->
    {
        matches: actual is Number,
        description: "Expected value to be Number type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value is of type Boolean
*
* === Example
*
* This example shows how to validate that a value is of type `Boolean`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* true must beBoolean()
* ----
*/
fun beBoolean(): Matcher = (actual: Any) ->
    {
        matches: actual is Boolean,
        description: "Expected value to be Boolean type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value is of type Object
*
* === Example
*
* This example shows how to validate that a value is of type `Object`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name : "Lionel", lastName: "Messi"} must beObject()
* ----
*/
fun beObject(): Matcher = (actual: Any) ->
    {
        matches: actual is Object,
        description: "Expected value to be Object type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value is of type Null
* === Example
*
* This example shows how to validate that a value is of type `Null`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* null must beNull()
* ----
*/
fun beNull(): Matcher = (actual: Any) ->
    {
        matches: actual is Null,
        description: "Expected value to be Null type but was $(typeOf(actual) as String)."
    }

/**
* Validates that a given value isn't of type Null
*
* === Example
*
* This example shows how to validate that a value is not of type `Null`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must notBeNull()
* ----
*/
fun notBeNull(): Matcher = (actual: Any) ->
    {
        matches: not (actual is Null),
        description: "Expected value not to be Null type."
    }

/**
* Validates that the asserted String contains the given String
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The expected text to be contained
* |===
*
* === Example
*
* Expects the value to contain the `String` "ex"
*
* ==== Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must contain("ex")
* ----
**/
fun contain(expected:String): Matcher<String> = (actual: String) ->
    {
        matches: actual contains expected,
        description: "Expected $(actual) to contain $(expected)."
    }

/**
* Validates that the asserted Array contains the given value
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The expected value to be contained
* |===
*
* === Example
*
* Expects the `Array` value to contain the `Number` 1
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, "A Text", true] must contain(1)
* ----
**/
fun contain(expected: Any): Matcher<Array<Any>> = (actual) ->
    {
        matches: actual contains expected,
        description: "Expected $(toReadableText(actual)) to contain $(toReadableText(expected))."
    }



/**
* Validates that the value is contained in the given Array
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The array of possible elements
* |===
*
* === Example
*
* Asserts that the value is either `1` or "A Text" or `true`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must beOneOf([1, "A Text", true])
* ----
**/
fun beOneOf(expected:Array<Any>): Matcher = (actual: Any) ->
    {
        matches: expected contains actual,
        description: "Expected value to be one of $(toReadableText(expected)) but was $(toReadableText(actual))."
    }

/**
* Validates that the asserted String ends with the given String
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | Suffix of the 'String'
* |===
*
* === Example
*
* Expects the `String` to end with "xt"
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must endWith("xt")
* ----
**/
fun endWith(expected:String): Matcher<String> = (actual:String) ->
    {
        matches: actual endsWith expected,
        description: "Expected $(actual) to end with $(expected)."
    }

/**
* Validates that the asserted String starts with the given String
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | Prefix of the 'String'
* |===
*
* === Example
*
* Validates that the `String` starts with "A"
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "A Text" must startWith("A")
* ----
**/
fun startWith(expected:String): Matcher<String> = (actual:String) ->
    {
        matches: actual startsWith expected,
        description: "Expected $(actual) to start with $(expected)."
    }

/**
* Validates that the value (String, Object or Array) is empty
*
*
* === Example
*
* Validates that the `Array` is of size `0`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [] must beEmpty()
* ----
*/
fun beEmpty(): Matcher<String | Object | Array | Null> = (actual: String | Object | Array | Null) ->
    {
        matches: actual is Null or isEmpty(actual default ""),
        description: "Expected $(toReadableText(actual)) to be empty."
    }

/**
* Validates that the String value is blank
*
* === Example
*
* Validates that the `String` is empty or has whitespaces
*
* ==== Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* "  " must beBlank()
* ----
*/
fun beBlank(): Matcher<String | Null> = (actual: String | Null) ->
    {
        matches: actual is Null or isBlank(actual default ""),
        description: "Expected $(actual default 'null') to be blank."
    }

/**
* Validates that the Object has the given key
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | keyName | The name of the key to expect to be present.
* |===
*
* === Example
*
* Validates that the `Object` contains a key called "name"
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must haveKey("name")
* ----
**/
fun haveKey(keyName: String): Matcher<Object> = (actual: Object) ->
    {
        matches: actual[keyName]?,
        description: "Expected $(toReadableText(actual)) to contain key $(keyName)."
    }

/**
* Validates that the Object has the given value
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value that is expected to be present
* |===
*
* === Example
*
* Expected that the `Object` contains the value "Messi"
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must haveValue("Messi")
* ----
**/
fun haveValue(value: Any): Matcher<Object> = (actual: Object) ->
    {
        matches: actual pluck $ contains value,
        description: "Expected $(toReadableText(actual)) to contain value $(toReadableText(value))."
    }

/**
* Validates that the given value is equal to the content of a resource file
*
* The resource file must belong to the classpath
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | resourceName | The resource name
* | contentType | The content type of the resource (Optional)
* | readerProperties | An object with the config properties (Optional)
* |===
*
* === Example
*
* Expects a value to be equal to the content of the resource "user.json"
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* { name: "Lionel", lastName: "Messi" } must equalToResource("user.json", "application/json")
* ----
**/
fun equalToResource(resourceName: String, contentType: String = "application/dw", readerProperties: Object = {}): Matcher<Any> =
    equalTo(readUrl("classpath://$(resourceName)", contentType, readerProperties))

/**
*  Validates that the asserted Comparable value is greater than the given one
*
*  Can be equal to when using the _inclusive_ argument
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The number to compare to
* | inclusive | If it is inclusive or not (Optional) false by default
* |===
*
* === Example
*
* Expects a `Number` to be bigger than `2`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 3 must beGreaterThan(2)
* ----
*
* === Example
*
* Expects a `Number` to be bigger or equal than `2`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 3 must beGreaterThan(2, true)
* ----
**/
fun beGreaterThan(expected: Comparable, inclusive: Boolean = false): Matcher<Comparable> = (actual: Comparable) ->
    {
        matches: if (inclusive) actual >= expected else actual > expected,
        description: "Expected value $(toReadableText(actual)) to be greater than $(expected)."
    }

/**
* Validates that the asserted Comparable value is lower than the given one
*
* Can be equal to when using the _inclusive_ argument
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | expected | The number to compare to
* | inclusive | If it is inclusive or not (Optional) false by default
* |===
*
* === Example
*
* Expects a `Number` to be less than `2`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must beLowerThan(2)
* ----
*
* === Example
*
* Expects a `Number` to be less or equal than `2`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* 1 must beLowerThan(2, true)
* ----
*
**/
fun beLowerThan(expected: Comparable, inclusive: Boolean = false): Matcher<Comparable> = (actual: Comparable) ->
    {
        matches: if (inclusive) actual <= expected else actual < expected,
        description: "Expected value $(toReadableText(actual)) to be lower than $(expected)."
    }


/**
* Validates that each item of the array satisfies the given matcher
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | matcher | The matcher to apply to all the elements
* |===
*
* === Example
*
* Expects all the elements in the `Array` to be a `Number`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1,2,3] must eachItem(beNumber())
* ----
**/
fun eachItem(matcher: Matcher<Any>): Matcher<Array<Any>> = (actual: Array<Any>) -> do {
    type IndexMatchPair = {index: Number, result: MatcherResult}

    fun collectAllFailed<T>(list: Array<T>, accumulator: Array<IndexMatchPair> = [], index: Number = 0): Array<IndexMatchPair> = do {
        list match {
            case [] -> accumulator
            case [head ~ tail] -> do {
                var matcherResult = matcher(head)
                ---
                if(matcherResult.matches)
                    collectAllFailed(tail, accumulator, index + 1)
                else
                    collectAllFailed(tail, accumulator + { index: index, result : matcherResult }, index + 1)
            }
        }
    }

    var allFailed = collectAllFailed(actual)
    ---
    {
        matches: isEmpty(allFailed),
        description: "Expected all items of the array to match but $(toReadableText(allFailed.index default [])) didn't match.",
        reasons: allFailed map (result) -> "Item $(result.index) $(result.result.description)."
    }
}

/**
* Validates that at least one item of the array satisfies the given matcher
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | matcher | The matcher to apply to at least one of the elements
* |===
*
* === Example
*
* Expects that one element of the `Array` is a `Number`
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::tests::Asserts
* ---
* [1, true, "a text"] must haveItem(beNumber())
* ----
**/
fun haveItem(matcher: Matcher<Any>): Matcher<Array<Any>> = (actual: Array<Any>) ->
    {
        matches: actual some matcher($).matches,
        description: "Expected any item of the array to match but none did."
    }
