package org.mule.weave.v2.module.avro

import org.apache.avro.generic.GenericRecord
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.AlreadyMaterializedObjectValue
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.ConfigurableSchemaSetting
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.DeferredWriter
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.SimpleLocation
import org.mule.weave.v2.parser.module.MimeType

class AvroDataFormat extends DataFormat[AvroReaderSettings, AvroWriterSettings] {

  override def name(): String = "avro"

  override val defaultMimeType: MimeType = MimeType("application", "avro")

  override val acceptedMimeTypes: Seq[MimeType] = Seq(defaultMimeType)

  override def reader(source: SourceProvider)(implicit ctx: EvaluationContext): Reader = new AvroReader(source, createReaderSettings())

  override def writer(target: Option[Any], outputMimeType: MimeType)(implicit ctx: EvaluationContext): DeferredWriter[AvroWriterSettings] = {
    DeferredWriter(AvroWriter.apply, TargetProvider(target), createWriterSettings())
  }

  override def readerSettings(): AvroReaderSettings = new AvroReaderSettings(this)

  override def writerSettings(): AvroWriterSettings = new AvroWriterSettings(this)

  override def fileExtensions: Seq[String] = Seq(".avro")
}

abstract class AvroSettings extends ConfigurableSchemaSetting {

  override val schemaExtension: String = "-avro.json"

  override def schemaUrlDescriptionUrl(): String = "data-format/avro/schemaUrl.asciidoc"

  override val propertyName: String = "schemaUrl"

}

class AvroReaderSettings(override val dataFormat: DataFormat[_, _]) extends AvroSettings {
}

class AvroWriterSettings(override val dataFormat: DataFormat[_, _]) extends AvroSettings with ConfigurableDeferred {
}

object AvroObjectValue {
  def apply(record: GenericRecord, locationPath: () => String): AvroObjectValue = new AvroObjectValue(new AvroObjectSeq(record), locationPath)
}

class AvroObjectValue(objectSeq: AvroObjectSeq, locationPath: () => String) extends AlreadyMaterializedObjectValue {
  override def evaluate(implicit ctx: EvaluationContext): T = {
    objectSeq
  }

  override def location(): Location = {
    SimpleLocation(locationPath())
  }
}
