package org.mule.weave.v2.module.avro

import org.apache.avro.AvroRuntimeException
import org.apache.avro.Conversion
import org.apache.avro.Conversions
import org.apache.avro.LogicalType
import org.apache.avro.LogicalTypes
import org.apache.avro.Schema
import org.apache.avro.Schema.Type
import org.apache.avro.file.DataFileWriter
import org.apache.avro.generic.GenericData
import org.apache.avro.generic.GenericData.EnumSymbol
import org.apache.avro.generic.GenericDatumWriter
import org.apache.avro.generic.GenericRecord
import org.mule.weave.v2.core.exception.MissingRequiredSettingException
import org.mule.weave.v2.core.exception.SchemaNotFoundException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.RangeType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.{ Type => WRType }
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.avro.exception.AvroInvalidTypeException
import org.mule.weave.v2.module.avro.exception.InvalidEnumValueException
import org.mule.weave.v2.module.avro.exception.InvalidFieldNameException
import org.mule.weave.v2.module.avro.exception.InvalidFieldValueException
import org.mule.weave.v2.module.avro.exception.InvalidValueForUnionException
import org.mule.weave.v2.module.avro.exception.InvalidValueLogicalTypeException
import org.mule.weave.v2.module.avro.exception.InvalidValueTypeException
import org.mule.weave.v2.module.avro.exception.MissingRequiredFieldException
import org.mule.weave.v2.module.avro.exception.UnableToConvertTypeException
import org.mule.weave.v2.module.option.ConfigurableSchemaSetting
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.location.Location

import java.io.ByteArrayInputStream
import java.io.OutputStream
import java.nio.ByteBuffer
import java.time.ZoneOffset
import java.util.{ HashMap => JMap }
import java.util.UUID
import scala.collection.JavaConverters._
import scala.collection.mutable.ArrayBuffer
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class AvroWriter(os: OutputStream, val settings: AvroSettings)(implicit ctx: EvaluationContext) extends Writer {

  override def flush(): Unit = {}

  override def close(): Unit = {}

  def writeObject(v: Value[ObjectSeq], schema: Schema)(implicit ctx: EvaluationContext): Any = {
    val objectSeq = v.evaluate
    schema.getType match {
      case Type.MAP => {
        val map = new JMap[String, Any]()
        objectSeq
          .toIterator()
          .foreach((kvp) => {
            val name = kvp._1.evaluate.name
            map.put(name, write(kvp._2, schema.getValueType))
          })
        map
      }
      case Type.RECORD => {
        val record = new GenericData.Record(schema)
        val processedFields = new Array[Schema.Field](schema.getFields.size())
        objectSeq
          .toIterator()
          .foreach((kvp) => {
            val name = kvp._1.evaluate.name
            val field: Schema.Field = schema.getField(name)
            if (field == null) {
              throw new InvalidFieldNameException(name, schema.getFields.asScala.map(_.name()), kvp._1.location())
            } else {
              try {
                val value: Any = write(kvp._2, field.schema())
                record.put(name, value)
                processedFields.update(field.pos(), field)
              } catch {
                case avroInvalidTypeException: AvroInvalidTypeException => throw new InvalidFieldValueException(name, avroInvalidTypeException.message, kvp._1.location())
              }
            }
          })

        schema.getFields.forEach((missingField) => {
          if (processedFields(missingField.pos()) == null) {
            if (missingField.defaultVal() != null) {
              record.put(missingField.pos(), missingField.defaultVal())
            } else {
              val fieldSchema = missingField.schema()
              throw new MissingRequiredFieldException(missingField.name(), fieldSchema.getType.name().toLowerCase, schema.getFullName, v.location())
            }
          }
        })

        record
      }
      case invalidType => {
        throw new InvalidValueTypeException(v.valueType, invalidType, v.location())
      }
    }
  }

  def writeArray(value: Value[ArraySeq], schema: Schema)(implicit ctx: EvaluationContext): GenericData.Array[Any] = {
    schema.getType match {
      case Type.ARRAY => {
        val evaluate = value.evaluate.toArray()
        val elementType = schema.getElementType
        val result = new GenericData.Array[Any](evaluate.length, schema)
        var i = 0
        while (i < evaluate.length) {
          result.add(write(evaluate(i), elementType))
          i = i + 1
        }
        result
      }
      case invalidType => {
        throw new InvalidValueTypeException(ArrayType, invalidType, value.location())
      }
    }

  }

  def write(value: Value[_], schema: Schema)(implicit ctx: EvaluationContext): Any = {
    schema.getType match {
      case Type.UNION => {
        val materializedValue = value.materialize
        val types = schema.getTypes
        var i = 0
        var exceptions = ArrayBuffer[Throwable]()
        var result: Option[Any] = None
        while (result.isEmpty && i < types.size()) {
          val writeResult = Try(write(materializedValue, types.get(i)))
          writeResult match {
            case Failure(exception) => {
              exceptions.+=(exception)
            }
            case Success(value) => {
              result = Some(value)
            }
          }
          i = i + 1
        }
        result match {
          case Some(v) => v
          case None    => throw new InvalidValueForUnionException(exceptions, value.location(), types.asScala.map(_.getType), value.valueType)
        }
      }
      case _ => {
        value match {
          case obj if ObjectType.accepts(obj)    => writeObject(ObjectType.coerce(value), schema)
          case array if ArrayType.accepts(array) => writeArray(ArrayType.coerce(value), schema)
          case range if RangeType.accepts(range) => writeArray(ArrayType.coerce(value), schema)
          case _ => {
            schema.getType match {
              case Type.LONG => {
                val maybeConversion: Option[Conversion[_]] = ConversionFactory.getConversion(schema.getLogicalType)
                maybeConversion match {
                  case Some(conversion) => {
                    val avroValue = schema.getLogicalType.getName match {
                      case x: String if (x.startsWith("timestamp")) => {
                        //Avro requires an Instance for timestap
                        value match {
                          case t if (DateTimeType.accepts(t)) => {
                            DateTimeType.coerce(value).evaluate.toInstant
                          }
                          case _ => {
                            LocalDateTimeType.coerce(value).evaluate.toInstant(ZoneOffset.UTC)
                          }
                        }
                      }
                      case _ => value.evaluate
                    }
                    convertToRawType(avroValue, schema, schema.getLogicalType, conversion, value.valueType, value.location())
                  }
                  case None => {
                    Try(NumberType.coerce(value).evaluate.toLong) match {
                      case Failure(_) => {
                        throw new InvalidValueTypeException(value.valueType, Type.LONG, value.location())
                      }
                      case Success(value) => value
                    }
                  }
                }
              }
              case Type.INT => {
                val maybeConversion: Option[Conversion[_]] = ConversionFactory.getConversion(schema.getLogicalType)
                maybeConversion match {
                  case Some(conversion) => {
                    convertToRawType(value.evaluate, schema, schema.getLogicalType, conversion, value.valueType, value.location())
                  }
                  case None => {
                    Try(NumberType.coerce(value).evaluate.toInt) match {
                      case Failure(_) => {
                        throw new InvalidValueTypeException(value.valueType, Type.INT, value.location())
                      }
                      case Success(value) => value
                    }
                  }
                }
              }
              case Type.FLOAT => {
                Try(NumberType.coerce(value).evaluate.toFloat) match {
                  case Failure(_) => {
                    throw new InvalidValueTypeException(value.valueType, Type.FLOAT, value.location())
                  }
                  case Success(value) => value
                }
              }
              case Type.DOUBLE => {
                Try(NumberType.coerce(value).evaluate.toDouble) match {
                  case Failure(_) => {
                    throw new InvalidValueTypeException(value.valueType, Type.DOUBLE, value.location())
                  }
                  case Success(value) => value
                }
              }
              case Type.BOOLEAN => {
                Try(BooleanType.coerce(value).evaluate) match {
                  case Failure(_) => {
                    throw new InvalidValueTypeException(value.valueType, Type.BOOLEAN, value.location())
                  }
                  case Success(value) => value
                }
              }
              case Type.STRING => {
                val maybeConversion: Option[Conversion[_]] = ConversionFactory.getConversion(schema.getLogicalType)
                maybeConversion match {
                  case Some(conversion) => {
                    schema.getLogicalType.getName match {
                      case x: String if x == LogicalTypes.uuid().getName => {
                        Try(StringType.coerce(value).evaluate) match {
                          case Failure(_) => {
                            throw new InvalidValueTypeException(value.valueType, Type.STRING, value.location())
                          }
                          case Success(evaluatedValue) => {
                            Try(UUID.fromString(evaluatedValue.toString)) match {
                              case Failure(_) => throw new InvalidValueLogicalTypeException(evaluatedValue.toString, LogicalTypes.uuid(), value.location())
                              case _          => evaluatedValue
                            }
                          }
                        }
                      }
                      case _ => convertToRawType(value.evaluate, schema, schema.getLogicalType, conversion, value.valueType, value.location())
                    }
                  }
                  case None => {
                    Try(StringType.coerce(value).evaluate) match {
                      case Failure(_) => {
                        throw new InvalidValueTypeException(value.valueType, Type.STRING, value.location())
                      }
                      case Success(value) => value
                    }
                  }
                }

              }
              case Type.BYTES => {
                val maybeConversion: Option[Conversion[_]] = ConversionFactory.getConversion(schema.getLogicalType)
                maybeConversion match {
                  case Some(conversion) => {
                    val bigDecimal: java.math.BigDecimal = NumberType.coerce(value).evaluate.toBigDecimal.bigDecimal
                    convertToRawType(bigDecimal, schema, schema.getLogicalType, conversion, value.valueType, value.location())
                  }
                  case _ => {
                    ByteBuffer.wrap(BinaryValue.getBytes(BinaryType.coerce(value)))
                  }
                }
              }
              case Type.ENUM => {
                Try(StringType.coerce(value).evaluate) match {
                  case Failure(_) => {
                    throw new InvalidValueTypeException(value.valueType, Type.ENUM, value.location())
                  }
                  case Success(value) if (schema.getEnumSymbols.contains(value)) => {
                    new EnumSymbol(schema, value)
                  }
                  case Success(strValue) => {
                    throw new InvalidEnumValueException(strValue.toString, schema.getEnumSymbols.toArray(new Array[String](0)), value.location())
                  }
                }
              }
              case Type.FIXED => {
                val maybeConversion: Option[Conversion[_]] = ConversionFactory.getConversion(schema.getLogicalType)
                maybeConversion match {
                  case Some(conversion) => {
                    val bigDecimal: java.math.BigDecimal = NumberType.coerce(value).evaluate.toBigDecimal.bigDecimal
                    convertToRawType(bigDecimal, schema, schema.getLogicalType, conversion, value.valueType, value.location())
                  }
                  case _ => {
                    val bytes = BinaryValue.getBytes(BinaryType.coerce(value))
                    new GenericData.Fixed(schema, bytes)
                  }
                }
              }
              case Type.NULL   => NullType.coerce(value).evaluate
              case invalidType => throw new InvalidValueTypeException(value.valueType, invalidType, value.location())
            }
          }
        }
      }
    }
  }

  def convertToRawType[T](datum: Any, schema: Schema, logicalType: LogicalType, conversion: Conversion[T], valueType: WRType, location: Location): AnyRef = {
    try {
      Conversions.convertToRawType(datum, schema, logicalType, conversion)
    } catch {
      case _: AvroRuntimeException => {
        throw new UnableToConvertTypeException(valueType, datum.getClass, conversion.getConvertedType, logicalType.getName, location)
      }
      case _: IllegalArgumentException => {
        throw new UnableToConvertTypeException(valueType, datum.getClass, conversion.getConvertedType, logicalType.getName, location)
      }
    }
  }

  protected override def doWriteValue(value: Value[_])(implicit ctx: EvaluationContext): Unit = {
    val schemaResult = settings.schema match {
      case Some(ConfigurableSchemaSetting.SchemaResult(resultSchema)) => resultSchema
      case Some(ConfigurableSchemaSetting.SchemaNotSet(_)) =>
        throw new MissingRequiredSettingException("schemaUrl", "Avro Writer", value.location())
      case Some(ConfigurableSchemaSetting.SchemaNotFound(_, schemaUrl)) =>
        throw new SchemaNotFoundException(value.location(), schemaUrl)
      case None =>
        throw new MissingRequiredSettingException("schemaUrl", "Avro Writer", value.location())
    }

    val schema: Schema = new Schema.Parser().parse(new ByteArrayInputStream(schemaResult))
    val datumWriter = new GenericDatumWriter[GenericRecord](schema)
    val dataFileWriter = new DataFileWriter[GenericRecord](datumWriter)

    dataFileWriter.create(schema, os)

    value match {
      case obj if ObjectType.accepts(obj) => {
        val result: Any = write(value, schema)
        result match {
          case gr: GenericRecord => dataFileWriter.append(gr)
          case _                 => throw new InvalidValueTypeException(value.valueType, schema.getType, value.location())
        }
      }
      case array if ArrayType.accepts(array) => {
        val iterator = ArrayType.coerce(value).evaluate.toIterator()

        iterator
          .foreach((value) => {
            val result: Any = write(value, schema)
            result match {
              case gr: GenericRecord => dataFileWriter.append(gr)
              case _                 => throw new InvalidValueTypeException(value.valueType, schema.getType, value.location())
            }
          })
      }
    }

    dataFileWriter.flush()
    dataFileWriter.close()

  }

  override def result: Any = {
    os
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(new AvroDataFormat)
}
object AvroWriter {
  def apply(tp: TargetProvider, settings: AvroSettings)(implicit ctx: EvaluationContext): AvroWriter = {
    new AvroWriter(tp.asOutputStream, settings)
  }
}
