/**
* An adapter between BAT IO module and DW IO module.
*/
%dw 2.0
import * from dw::Runtime
import * from dw::core::URL
import * from dw::io::http::Client
import * from dw::io::http::Types
import * from dw::io::http::utils::HttpHeaders
import mergeWith from dw::core::Objects
import normalizeHeaders from dw::io::http::BodyUtils
import dw::util::Coercions
import dw::module::Mime

var BAT_DEFAULT_HTTP_REQUEST_CONFIG = {
  followRedirect: false,
  readTimeout: 30000
}

var BAT_DEFAULT_HTTP_CLIENT_CONFIG = {
  connectionTimeout: 10000,
  compressionHeader: true,
  decompress: true,
  tls: {
    insecure: false
  }
}

var DEFAULT_IO_HEADERS = {
  '$(CONNECTION_HEADER)': "close",
  '$(USER_AGENT_HEADER)': "DataWeave/2.0"
}

fun createHttpRequestConfig(options: dw::http::Types::HttpClientOptionalOptions): HttpRequestConfig = do {
  var config = BAT_DEFAULT_HTTP_REQUEST_CONFIG
    mergeWith {
      (followRedirect: options.allowRedirect!) if (options.allowRedirect?),
      (readTimeout: options.readTimeout!) if (options.readTimeout?)
    }
  ---
  config
}

fun createHttpClientConfig(options: dw::http::Types::HttpClientOptionalOptions): HttpClientConfig = do {
  var config = BAT_DEFAULT_HTTP_CLIENT_CONFIG
    mergeWith {
      (connectionTimeout: options.connectionTimeout!) if (options.connectionTimeout?),
      (compressionHeader: options.allowCompression!) if (options.allowCompression?),
      (decompress: options.allowCompression!) if (options.allowCompression?)
    }
  // Add TLS configurations
  var tls = if (options.allowUnsafeSSL?)
    config mergeWith {
      tls: { insecure: options.allowUnsafeSSL!}
    }
    else config
  ---
  tls
}

fun createHARTimers(httpResponse: HttpResponse<Binary, Any>): dw::http::Types::HARTimers = do {
  var schema = httpResponse.^ default {}
  var timers = schema.timers default {}
  ---
  {
    (dns: timers.dns! as Number) if (timers.dns?),
    (connect: timers.connect! as Number) if (timers.connect?),
    (ssl: timers.tls! as Number) if (timers.tls?),
    (send: timers.send! as Number) if (timers.send?),
    (wait: timers.wait! as Number) if (timers.wait?),
    (receive: timers.receive! as Number) if (timers.receive?),
    total: schema.total as Number default 0
  }
}

fun createHttpClientResult(httpRequest: HttpRequest<Binary>, uri: URI, httpResponse: TryResult<HttpResponse<Binary, Any>>, bodyMapper: (HttpResponse<Binary, Any>) -> Any | Null, options: Object): dw::http::Types::HttpClientResult = do {
  var result = httpResponse.result
  var httpClientResult =
  if (result != null)
    {
      err: false,
      options: options,
      request: createHttpClientRequest(httpRequest, uri),
      timers: createHARTimers(result),
      response: createHttpClientResponse(result, bodyMapper)
    }
  else
    {
      err: true,
      message: httpResponse.error.message
    }
  ---
  httpClientResult
}

fun createHttpClientResponse(httpResponse: HttpResponse<Binary, Any>, bodyMapper: (HttpResponse<Binary, Any>) -> Any | Null): dw::http::Types::HttpClientResponse<dw::http::Types::BodyType, dw::http::Types::HttpStrictHeaders> = do {
  var body = bodyMapper(httpResponse)
  var mimeType = if (httpResponse.contentType?)
    do {
      var contentType = httpResponse.contentType!
      var mime = Mime::fromString(contentType)
      ---
      if (mime.success)
        "$(mime.result.'type')/$(mime.result.subtype)"
      else
        null
    }
    else
      null
  var normalizedHeaders = normalizeHeaders(httpResponse.headers)
  var response = {
    status: httpResponse.status,
    statusText: httpResponse.statusText onNull(() -> ""),
    headers: toHttpStrictHeaders(normalizedHeaders),
    (payload: httpResponse.body!) if (httpResponse.body?),
    (body: body) if (body != null),
    cookies: toHttpResponseCookies(httpResponse.cookies),
    (contentType: httpResponse.contentType!) if (httpResponse.contentType?),
    (mime: mimeType as String) if (mimeType != null)
  }
  ---
  response
}

fun toHttpStrictHeaders(headers: HttpHeaders): dw::http::Types::HttpStrictHeaders = do {
  headers mapObject ((value, key, index) -> (key): Coercions::toString(value))
}

fun resolveRequestPath(uri: URI): String =
  if (uri.isValid)
    uri.path onNull (() -> "")
  else
    ""

fun resolvedRequestPort(uri: URI): Number = do {
  if (uri.isValid)
    uri.port onNull (() -> do {
      if (uri.scheme != null) do {
        uri.scheme match {
          case "https" -> 443
          else -> 80
        }
      } else -1
    })
  else
    -1
}

fun toHttpResponseCookies(cookies: HttpResponseCookies): dw::http::Types::HttpResponseCookies = do {
  cookies mapObject ((cookie, key, index) -> (key): {
    name: cookie.name,
    value: cookie.value,
    (domain: cookie.domain!) if (cookie.domain?),
    (comment: cookie.comment!) if (cookie.comment?),
    (path: cookie.path!) if (cookie.path?),
    maxAge: cookie.maxAge,
    httpOnly: cookie.httpOnly,
    secure: cookie.secure
  })
}

fun createHttpClientRequest(httpRequest: HttpRequest<Binary>, uri: URI): dw::http::Types::HttpClientRequest = do {
  var httpClientRequest = {
    httpVersion: "",
    url: httpRequest.url as String,
    path: resolveRequestPath(uri),
    method: httpRequest.method,
    ip: "",
    port: resolvedRequestPort(uri),
    (headers: toHttpStrictHeaders(httpRequest.headers!)) if (httpRequest.headers?),
    (payload: httpRequest.body!) if (httpRequest.body?)
  }
  ---
  httpClientRequest
}

fun populateBasicHeaders(clientOptions: dw::http::Types::HttpClientOptionalOptions, uri: URI): HttpHeaders = do {
  var headers = clientOptions.headers default {}
  var hostHeader = if (uri.isValid) do {
    var resolvedPort = resolvedRequestPort(uri)
    var port = if (resolvedPort == -1 or resolvedPort == 80 or resolvedPort == 443) "" else ":$(resolvedPort)"
    ---
    { ('$(HOST_HEADER)': "$(uri.host!)$(port)") if (uri.host?) }
  }
  else {}
  ---
  DEFAULT_IO_HEADERS
    mergeWith
      hostHeader
    mergeWith
      headers
}

fun nativeRequestAdapter(clientOptions: dw::http::Types::HttpClientOptions): dw::http::Types::HttpClientResult = do {
  var uri = parseURI(clientOptions.url as String)
  var headers = populateBasicHeaders(clientOptions, uri)
  var request = {
    method: clientOptions.method as HttpMethod,
    url: clientOptions.url,
    headers: headers,
    (body: clientOptions.body! as Binary) if (clientOptions.body?)
  }
  var requestConfig  = createHttpRequestConfig(clientOptions)
  var clientConfig = createHttpClientConfig(clientOptions)
  var maybeHttpResponse = try(() -> sendRequest(request, requestConfig, clientConfig))
  var httpClientResult = createHttpClientResult(request, uri, maybeHttpResponse, (response) -> null, clientOptions)
  ---
  httpClientResult
}

fun requestAdapter(clientOptions: dw::http::Types::HttpClientOptions,
  serializationOptions: {readerOptions?: Object, writerOptions?: Object },
  options: Object): dw::http::Types::HttpClientResult = do {

  var writerProperties = serializationOptions.writerOptions default {}

  var uri = parseURI(clientOptions.url as String)

  var basicHeaders  = populateBasicHeaders(clientOptions, uri)

  var normalizedHeaders = normalizeHeaders(basicHeaders)
  var contentType = normalizedHeaders[CONTENT_TYPE_HEADER] default (
    clientOptions.body match {
      case is Binary -> "application/octet-stream"
      case is dw::module::Multipart::Multipart -> "multipart/form-data; boundary=$(writerProperties.boundary default dw::module::Multipart::generateBoundary())"
      case is String -> "text/plain"
      else -> "application/json"
    })

  // Update 'Content-Type' header (using normalized headers to avoid Content-Type header duplication)
  var headers = normalizedHeaders
    update {
      case ."$(CONTENT_TYPE_HEADER)"! -> contentType
    }

  var request = {
    method: clientOptions.method as HttpMethod,
    url: clientOptions.url,
    headers: headers,
    (body: clientOptions.body!) if (clientOptions.body?)
  }

  var requestConfig = createHttpRequestConfig(clientOptions)

  var serializationConfig = {
    contentType: "application/json",
    (readerProperties: serializationOptions.readerOptions!) if (serializationOptions.readerOptions?),
    (writerProperties: serializationOptions.writerOptions!) if (serializationOptions.writerOptions?)
  }

  var clientConfig = createHttpClientConfig(options)

  var binaryRequest = createBinaryHttpRequest(request, serializationConfig)

  // Set 'Content-Length' header
  var updatedBinaryRequest = (binaryRequest update {
    case headers at .headers -> headers mergeWith {
      ('$(CONTENT_LENGTH_HEADER)': sizeOf(binaryRequest.body)) if (binaryRequest.body != null)
    }
  }) as HttpRequest<Binary>

  var maybeHttpResponse = try(() -> sendRequest(updatedBinaryRequest, requestConfig, clientConfig))

  var httpClientResult= createHttpClientResult(updatedBinaryRequest, uri, maybeHttpResponse, (response) -> do {
    var responseBody = response.body
    ---
    if (responseBody != null) do {
      var contentType = response.contentType
      ---
      if (contentType != null)
        try(() -> readBody(contentType, responseBody, serializationConfig.readerProperties default {})).result
      else
        responseBody
    } else
      responseBody
  }, options)
  ---
  httpClientResult
}