package com.mulesoft.weave.compiler

import com.mulesoft.weave.compiler.WeaveBinaryCompiler.getClass
import org.mule.weave.v2.compilation.mapper.{AstToSerializableMapper, ResourceBasedSerializerContext}
import org.mule.weave.v2.compilation.serializer.SerializableAstNodeSerializer
import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.module.pojo.ClassLoaderServiceAware
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.module.ModuleLoaderProvider
import org.mule.weave.v2.parser.phase.{ModuleLoader, ModuleLoaderManager, ModuleParsingPhasesManager, ParsingContext}
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk._
import org.mule.weave.v2.utils.WeaveFile

import java.io._
import java.lang.String.format
import java.util
import java.util.ServiceLoader
import scala.collection.JavaConverters.collectionAsScalaIterable

object WeaveBinaryCompiler {

  def toBinary(resourceDirectory: File, outputBaseDirectory: File, maybeClassLoader: Option[ClassLoader] = Option.empty) = {
    val folderBasedResourceResolver = FolderBasedResourceResolver(resourceDirectory)
    val classLoaderWeaveResourceResolver = maybeClassLoader.map(classLoader => ClassLoaderWeaveResourceResolver.providedClassLoader(Seq.apply(classLoader))).getOrElse(ClassLoaderWeaveResourceResolver.apply())
    val loaders = Seq(ModuleLoader(classLoaderWeaveResourceResolver), ModuleLoader(folderBasedResourceResolver))
    val moduleLoaderProvider = new EnricherModuleLoaderProvider(maybeClassLoader.getOrElse(getClass.getClassLoader), folderBasedResourceResolver)
    val moduleParsingPhasesManager: ModuleParsingPhasesManager = ModuleParsingPhasesManager.apply(ModuleLoaderManager.apply(loaders, moduleLoaderProvider))
    val runtimeModuleNodeCompiler = RuntimeModuleNodeCompiler(moduleParsingPhasesManager)

    compile(file = resourceDirectory, resourceDirectory = resourceDirectory, outputBaseDirectory = outputBaseDirectory, moduleParsingPhasesManager = moduleParsingPhasesManager, runtimeModuleNodeCompiler = runtimeModuleNodeCompiler)
  }

  private class EnricherModuleLoaderProvider(classLoader: ClassLoader, weaveResourceResolver: WeaveResourceResolver) extends ModuleLoaderProvider {

    override def getModules: Seq[ModuleLoader] = {
      val iterator: util.Iterator[ModuleLoader] = ServiceLoader.load(classOf[ModuleLoader], classLoader).iterator
      val modules: util.ArrayList[ModuleLoader] = new util.ArrayList[ModuleLoader]
      while (iterator.hasNext) {
        val next: ModuleLoader = iterator.next
        if (next.isInstanceOf[WeaveResourceResolverAware]) next.asInstanceOf[WeaveResourceResolverAware].resolver(weaveResourceResolver)
        if (next.isInstanceOf[ClassLoaderServiceAware]) next.asInstanceOf[ClassLoaderServiceAware].classLoaderService((className: String) => {
          try Option.apply(classLoader.loadClass(className))
          catch {
            case e: ClassNotFoundException =>
              Option.empty
          }

        })
        modules.add(next)
      }
      collectionAsScalaIterable(modules).toSeq
    }
  }

  private def compile(name: String = "", file: File, resourceDirectory: File, outputBaseDirectory: File, moduleParsingPhasesManager: ModuleParsingPhasesManager, runtimeModuleNodeCompiler: RuntimeModuleNodeCompiler): Unit = {
    val files = file.listFiles()
    if (files != null) {
      files.foreach(file => {
        if (file.getName.endsWith(WeaveFile.fileExtension)) {
          println(format("Compiling %s", resourceDirectory.toPath.relativize(file.toPath)))
          val basename = FileHelper.baseName(file.getName)
          val identifier = NameIdentifier.apply(nameIdentifier(name, basename), Option.empty)
          val parsingContext = ParsingContextFactory.createParsingContext(identifier, moduleParsingPhasesManager)

          val weaveResource = WeaveResourceFactory.fromFile(file)
          val documentParser = new DocumentParser
          val parseResult = documentParser.parse(weaveResource, parsingContext)
          if (!parseResult.hasResult() || parseResult.hasErrors()) {
            println(format("Errors reported by parser for %s", file.getAbsolutePath))
            parseResult.errorMessages().foreach((messages) => {
              println(messages._2.message + "\nat \n" + messages._1.locationString)
            })
            throw new RuntimeException("File has compilation errors, it cannot be compiled to binary")
          } else {
            val result = parseResult.getResult()
            binaryCompile(file, resourceDirectory, outputBaseDirectory, result.astNode, weaveResource, parsingContext)
          }
        } else {
          if (file.isDirectory) {
            compile(nameIdentifier(name, file.getName), file, resourceDirectory, outputBaseDirectory, moduleParsingPhasesManager, runtimeModuleNodeCompiler)
          }
        }
      })
    }
  }

  private def nameIdentifier(name: String, moduleName: String): String = {
    if (name.isEmpty) {
      moduleName
    } else {
      name + NameIdentifier.SEPARATOR + moduleName
    }
  }

  private def binaryCompile(file: File, resourceDirectory: File, outputBaseDirectory: File, parserAstNode: AstNode, weaveResource: WeaveResource, parsingContext: ParsingContext) = {
    val preCompilationResultPhaseResult = parserAstNode match {
      case _: ModuleNode =>
        WeaveCompiler.preCompileModule(
          weaveResource,
          parsingContext)
      case _: DocumentNode =>
        WeaveCompiler.preCompile(
          weaveResource,
          parsingContext)
    }

    if (!preCompilationResultPhaseResult.hasErrors() && preCompilationResultPhaseResult.hasResult()) {
      val astNode = preCompilationResultPhaseResult.getResult().astNode
      toBinary(astNode, file, resourceDirectory, outputBaseDirectory)
    } else {
      println(s"Errors reported by parser for ${weaveResource.url()}")
      preCompilationResultPhaseResult.errorMessages().foreach((messages) => {
        println(messages._2.message + "\nat \n" + messages._1.locationString)
      })
      throw new RuntimeException("File has compilation errors, it cannot be compiled to binary")
    }
  }

  def toBinary(astNode: AstNode, file: File, resourceDirectory: File, outputBaseDirectory: File): File = {
    val relativeWeaveFile = resourceDirectory.toPath.relativize(file.toPath)

    val serializableAstNode = AstToSerializableMapper.serialize(astNode)

    val outputPath = outputBaseDirectory.toPath.resolve(relativeWeaveFile)

    val binaryFile = new File(
      outputPath.getParent.toFile,
      file.getName.replace(WeaveFile.fileExtension, WeaveFile.binaryFileExtension))
    binaryFile.getParentFile.mkdirs()

    val dataOutputStream: DataOutputStream = new DataOutputStream(new FileOutputStream(binaryFile))
    try {
      SerializableAstNodeSerializer.serialize(serializableAstNode, dataOutputStream)
      binaryFile
    } finally {
      try {
        dataOutputStream.close()
      } catch {
        case _: Throwable => // Just ignore
      }
    }
  }

  def toAstNode(binaryFile: File, sourceWeaveResource: WeaveResource, nameIdentifier: NameIdentifier): AstNode = {
    val dataInputStream: DataInputStream = new DataInputStream(new FileInputStream(binaryFile))
    try {
      def node = SerializableAstNodeSerializer.deserialize(dataInputStream)
      AstToSerializableMapper.deserialize(node, ResourceBasedSerializerContext(sourceWeaveResource, nameIdentifier))
    } finally {
      try {
        dataInputStream.close()
      } catch {
        case _: Throwable => // Just ignore
      }
    }
  }

  private case class FolderBasedResourceResolver(sourceFolder: File) extends WeaveResourceResolver {

    override def resolve(name: NameIdentifier): Option[WeaveResource] = {
      val weaveFilePath = NameIdentifierHelper.toWeaveFilePath(name, "/")
      resolvePath(weaveFilePath);
    }

    override def resolvePath(path: String): Option[WeaveResource] = {
      val weaveFilePath = if (path.startsWith("/")) path.substring(1) else path
      val resource = new File(sourceFolder, weaveFilePath)
      if (resource.exists()) {
        Option.apply(WeaveResourceFactory.fromFile(resource))
      } else {
        Option.empty
      }
    }
  }

}
