/**
 * This helper module provide functions for creating MultiPart and
 * formats and parts (including fields and boundaries) of MultiPart formats.
 *
 *
 * To use this module, you must import it into your DataWeave code, for example,
 * by adding the line `import dw::module::Multipart` to the header of your
 * DataWeave script.
 */
%dw 2.0

/**
* `MultipartPart` type, a data structure for a part within a `MultiPart` format.
* See the output examples for the Multipart `field` function
* https://docs.mulesoft.com/dataweave/latest/dw-multipart-functions-field[documentation].
*/
type MultipartPart = {
  headers?: {
    "Content-Disposition"?: {
      name: String,
      filename?: String
    },
    "Content-Type"?: String
  },
  content: Any
}

/**
* `MultiPart` type, a data structure for a complete `Multipart` format. See the
* output example for the Multipart `form` function
* https://docs.mulesoft.com/dataweave/latest/dw-multipart-functions-form[documentation].
*/
type Multipart = {
  preamble?: String,
  parts: {
    _?: MultipartPart
  }
}

/**
* Creates a `MultipartPart` data structure using the specified part name,
* input content for the part, format (or mime type), and optionally, file name.
*
*
* This version of the `field` function accepts arguments as an array of objects
* that use the parameter names as keys, for example:
* `Multipart::field({name:"order",value: myOrder, mime: "application/json", fileName: "order.json"})`
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | opts | Array of objects that specifies:
*
* * A unique `name` (required) for the `Content-Disposition` header of the part.
* * A `value` (required) for the content of the part.
* * `mime` (optional for strings) for the mime type (for example, `application/json`) to apply to content within the part. This setting can be used to transform the input type, for example, from JSON to XML.
* * An optional `fileName` value that you can supply to the `filename` parameter in the part's `Content-Disposition` header.
* |===
*
* === Example
*
* This example produces two parts. The first part (named `order`) outputs
* content in JSON and provides a file name for the part (`order.json`). The
* second (named `clients`) outputs content in XML and does not provide a file
* name. Also notice that in this example you need to add the function's
* namespace to the function name, for example, `Multipart::field`.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import dw::module::Multipart
* output multipart/form-data
* var myOrder = [
*   {
*     order: 1,
*     amount: 2
*   },
*   {
*     order: 32,
*     amount: 1
*   }
* ]
* var myClients = {
*   clients: {
*     client: {
*       id: 1,
*       name: "Mariano"
*     },
*     client: {
*       id: 2,
*       name: "Shoki"
*     }
*   }
* }
* ---
* {
*   parts: {
*     order: Multipart::field({name:"order",value: myOrder, mime: "application/json", fileName: "order.json"}),
*     clients: Multipart::field({name:"clients", value: myClients, mime: "application/xml"})
*   }
* }
* ----
*
* ==== Output
*
* [source,txt,linenums]
* ----
* ------=_Part_8032_681891620.1542560124825
* Content-Type: application/json
* Content-Disposition: form-data; name="order"; filename="order.json"
*
* [
*   {
*     "order": 1,
*     "amount": 2
*   },
*   {
*     "order": 32,
*     "amount": 1
*   }
* ]
* ------=_Part_8032_681891620.1542560124825
* Content-Type: application/xml
* Content-Disposition: form-data; name="clients"
*
* <clients>
*   <client>
*     <id>1</id>
*     <name>Mariano</name>
*   </client>
*   <client>
*     <id>2</id>
*     <name>Shoki</name>
*   </client>
* </clients>
* ------=_Part_8032_681891620.1542560124825--
* ----
*/
fun field(opts: {|name: String, value: Any, mime?: String, fileName?: String |}): MultipartPart =  do {
    field(opts.name, opts.value, opts.mime default '', opts.fileName default '')
}

/**
* Creates a `MultipartPart` data structure using the specified part name,
* input content for the part, format (or mime type), and optionally, file name.
*
*
* This version of the `field` function accepts arguments in a comma-separated
* list, for example:
*
* [source,txt,linenums]
* ----
* Multipart::field("order", myOrder,"application/json", "order.json")`
* ----
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | opts | A set of parameters that specify:
*
* * A unique `name` (required) for the `Content-Disposition` header of the part.
* * A `value` (required) for the content of the part.
* * `mime` (optional for strings) for the mime type (for example, `application/json`) to apply to content within the part. This type can be used to transform the input type.
* * An optional `fileName` value that you can supply to the `filename` parameter in the part's `Content-Disposition` header.
* |===
*
* === Example
*
* This example produces two parts. The first part (named `order`) outputs
* content in JSON and provides a file name for the part (`order.json`). The
* second (named `clients`) outputs content in XML and does not provide a file
* name. The only difference between this `field` example and the previous
* `field` example is the way you pass in arguments to the method. Also notice
* that in this example you need to add the function's namespace to the function
* name, for example, `Multipart::field`.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import dw::module::Multipart
* output multipart/form-data
* var myOrder = [
*   {
*     order: 1,
*     amount: 2
*   },
*   {
*     order: 32,
*     amount: 1
*   }
* ]
* var myClients = {
*   clients: {
*     client: {
*       id: 1,
*       name: "Mariano"
*     },
*     client: {
*       id: 2,
*       name: "Shoki"
*     }
*   }
* }
* ---
* {
*   parts: {
*     order: Multipart::field("order", myOrder, "application/json", "order.json"),
*     clients: Multipart::field("clients", myClients, "application/xml")
*   }
* }
* ----
*
* ==== Output
*
* [source,txt,linenums]
* ----
* ------=_Part_4846_2022598837.1542560230901
* Content-Type: application/json
* Content-Disposition: form-data; name="order"; filename="order.json"
*
* [
*   {
*     "order": 1,
*     "amount": 2
*   },
*   {
*     "order": 32,
*     "amount": 1
*   }
* ]
* ------=_Part_4846_2022598837.1542560230901
* Content-Type: application/xml
* Content-Disposition: form-data; name="clients"
*
* <clients>
*   <client>
*     <id>1</id>
*     <name>Mariano</name>
*   </client>
*   <client>
*     <id>2</id>
*     <name>Shoki</name>
*   </client>
* </clients>
* ------=_Part_4846_2022598837.1542560230901--
* ----
*/
fun field(name: String, value: Any, mime: String = "", fileName: String = ""): MultipartPart =
  {
    headers: {
      ("Content-Type": mime) if mime != '',
      "Content-Disposition": {
        "name": name,
        ("filename": fileName) if fileName != ''
      }
    },
    content: value
  }

/**
* Creates a `MultipartPart` data structure from a resource file.
*
*
* This version of the `file` function accepts arguments as an array of objects
* that use the parameter names as keys, for example:
*
* [source,txt,linenums]
* ----
* Multipart::file({ name: "myFile", path: "myClients.json", mime: "application/json", fileName: "partMyClients.json"})
* ----
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | opts | Array of objects that specifies:
*
* * A unique `name` (required) for the `Content-Disposition` header of the part.
* * A `path` (required) relative to the `src/main/resources` project path for the Mule app.
* * `mime` (optional for strings) for the mime type (for example, `application/json`) to apply to content within the part. This setting _cannot_ be used to transform the input mime type.
* * An optional `fileName` value for the `filename` parameter in the part's `Content-Disposition` header. Defaults to the string `"filename"` if not
supplied. This value does not need to match the input file name.
* |===
*
* === Example
*
* This example inserts file content from a `MultipartPart` into a `Multipart`
* data structure. It uses the `form` function to create the `Multipart`
* and uses `file` to create a part named `myClient` with JSON content from
* an external file `myClients.json`. It also specifies `partMyClients.json` as
* the value for to the `filename` parameter.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import dw::module::Multipart
* output multipart/form-data
* var myClients = "myClients.json"
* var myArgs = { name: "myFile", path: "myClients.json", mime: "application/json", * fileName: "partMyClients.json"}
* ---
* Multipart::form([
*   Multipart::file(myArgs)
* ])
* ----
*
* ==== Input
*
* A file called `myClients.json` and located in `src/main/resources` with the
* following content.
*
* [source,json,linenums]
* ----
* clients: {
*   client: {
*     id: 1,
*      name: "Mariano"
*    },
*    client: {
*      id: 2,
*      name: "Shoki"
*   }
* }
* ----
*
* ==== Output
*
* [source,txt,linenums]
* ----
* ------=_Part_1586_1887987980.1542569342438
* Content-Type: application/json
* Content-Disposition: form-data; name="myFile"; filename="partMyClients.json"
*
* {
*    clients: {
*      client: {
*        id: 1,
*        name: "Mariano"
*      },
*      client: {
*        id: 2,
*        name: "Shoki"
*      }
*    }
* }
* ------=_Part_1586_1887987980.1542569342438--
* ----
*/
fun file(opts: {| name: String, path: String, mime?: String, fileName?: String |}) =
    file(opts.name, opts.path, opts.mime default 'application/octet-stream', opts.fileName default 'filename')

/**
* Creates a `MultipartPart` data structure from a resource file.
*
*
* This version of the `file` function accepts arguments in a comma-separated
* list, for example:
*
* [source,txt,linenums]
* ----
* Multipart::field("myFile", myClients, 'application/json', "partMyClients.json")
* ----
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | opts | Array of objects that specifies:
*
* * A unique `name` (required) for the `Content-Disposition` header of the part.
* * A `path` (required) relative to the `src/main/resources` project path for the Mule app.
* * `mime` (optional for strings) for the mime type (for example, `application/json`) to apply to content within the part. This setting _cannot_ be used to transform the input mime type.
* * An optional `fileName` value for the `filename` parameter in the part's `Content-Disposition` header. Defaults to the string `"filename"` if not
supplied. This value does not need to match the input file name.
* |===
*
* === Example
*
* This example inserts file content from a `MultipartPart` into a `Multipart`
* data structure. It uses the `form` function to create the `Multipart` type
* and uses `file` to create a part named `myClient` with JSON content from
* an external file `myClients.json`. It also specifies `partMyClients.json` as
* the value for to the `filename` parameter.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import dw::module::Multipart
* var myClients = "myClients.json"
* output multipart/form-data
* ---
* Multipart::form([
*  Multipart::file("myFile", myClients, 'application/json', "partMyClients.json")
* ])
* ----
*
* ==== Input
*
* A file called `myClients.json` and located in `src/main/resources` with the
* following content.
*
* [source,JSON,linenums]
* ----
* clients: {
*     client: {
*       id: 1,
*       name: "Mariano"
*     },
*     client: {
*       id: 2,
*       name: "Shoki"
*     }
*   }
* ----
*
* ==== Output
*
* [source,txt,linenums]
* ----
* ------=_Part_1586_1887987980.1542569342438
* Content-Type: application/json
* Content-Disposition: form-data; name="myFile"; filename="partMyClients.json"
*
* {
*    clients: {
*      client: {
*        id: 1,
*        name: "Mariano"
*      },
*      client: {
*        id: 2,
*        name: "Shoki"
*      }
*    }
* }
* ------=_Part_1586_1887987980.1542569342438--
* ----
*/
fun file(fieldName: String, path: String, mime: String = 'application/octet-stream', sentFileName: String = 'filename') =
  field(fieldName, readUrl('classpath://$(path)', 'application/octet-stream') as Binary, mime, sentFileName)

/**
* Creates a `Multipart` data structure using a specified array of parts.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | parts | An array of parts (`MultipartPart` data structures).
* |===
*
* === Example
*
* This example creates a `Multipart` data structure that contains parts, which
* are described in examples for the `field` function. For additional uses of
* `form`, see examples in the Multipart `file` and `field`
* https://docs.mulesoft.com/dataweave/latest/dw-multipart-functions-form[documentation].
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import dw::module::Multipart
* output multipart/form-data
* var firstPart = "content for my first part"
* var secondPart = "content for my second part"
* ---
* {
*   parts: {
*     part1: Multipart::field({name:"myFirstPart",value: firstPart}),
*     part2: Multipart::field("mySecondPart", secondPart)
*   }
* }
* ----
*
* ==== Output
*
* [source,txt,linenums]
* ----
* ------=_Part_320_1528378161.1542639222352
* Content-Disposition: form-data; name="myFirstPart"
*
* content for my first part
* ------=_Part_320_1528378161.1542639222352
* Content-Disposition: form-data; name="mySecondPart"
*
* content for my second part
* ------=_Part_320_1528378161.1542639222352--
* ----
*/
fun form(parts: Array<MultipartPart>): Multipart =
  {
    parts: parts reduce ((val, carry = {}) ->
      carry ++
      { (val.headers['Content-Disposition'].name): val }
    )
  }


/**
* Helper function for generating boundaries in `Multipart` data structures.
*/
fun generateBoundary(len: Number = 70): String = do {
    /**
    * Default Boundary separator
    */
    var boundaryChars = "-_1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
    var maxChars = sizeOf(boundaryChars)
    ---
    (1 to len) map boundaryChars[floor(random() * maxChars)] joinBy ''
}
