package com.mulesoft.weave.docs

import org.asciidoctor.Asciidoctor
import org.asciidoctor.Asciidoctor.Factory.create
import org.asciidoctor.Attributes
import org.asciidoctor.Options
import org.asciidoctor.SafeMode
import org.asciidoctor.jruby.AsciiDocDirectoryWalker
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.utils.AsciiDocMigrator

import java.io._
import java.nio.file.Files
import scala.collection.JavaConverters._
import scala.util.Failure
import scala.util.Try

object DocTemplateBundles {

  val asciidoc: DocTemplateBundle = DocTemplateBundle(
    Seq(
      DocTemplate(ClasspathDocResource("templates/asciidoc/asciidoc.vm"))),
    tableOfContent = Some(DocTemplate(ClasspathDocResource("templates/asciidoc/index.vm"), outputFileNamePattern = "index")),
    postProcessor = Some(new HtmlPostProcessor()),
    targetFolder = "asciidocs")

  val muledocs: DocTemplateBundle = DocTemplateBundle(
    Seq(
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/function.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}-functions-${utils.toValidFileName($function.name())}", selector = TemplateSelector.FUNCTION),
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/module.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}", TemplateSelector.MODULE),
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/namespaces.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}-namespaces", TemplateSelector.NAMESPACES),
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/types.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}-types", TemplateSelector.TYPES),
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/annotations.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}-annotations", TemplateSelector.ANNOTATIONS),
      DocTemplate(ClasspathDocResource("templates/asciidoc_2/variables.vm"), ".adoc", packageFolder = false, "dw-${module_name.toLowerCase()}-variables", TemplateSelector.VARIABLES)),
    tableOfContent = Some(DocTemplate(ClasspathDocResource("templates/asciidoc_2/index.vm"), outputFileNamePattern = "nav-dw", targetFolder = Some("_partials"))),
    targetFolder = "mule_docs")

  val markdown: DocTemplateBundle = DocTemplateBundle(
    Seq(
      DocTemplate(ClasspathDocResource("templates/markdown/body.vm"))),
    tableOfContent = Some(DocTemplate(ClasspathDocResource("templates/markdown/index.vm"), outputFileNamePattern = "home")),
    mappings = Some(DocTemplate(ClasspathDocResource("templates/markdown/mappings.vm"), outputFileNamePattern = "Mappings")),
    postProcessor = Some(new MarkdownPostProcessor("markdown")),
    targetFolder = "markdown_asciidocs")

  val exchangeMarkdown: DocTemplateBundle = DocTemplateBundle(
    Seq(DocTemplate(ClasspathDocResource("templates/exchange_markdown/body.vm"))),
    tableOfContent = Some(DocTemplate(ClasspathDocResource("templates/exchange_markdown/index.vm"), outputFileNamePattern = "home")),
    mappings = Some(DocTemplate(ClasspathDocResource("templates/exchange_markdown/mappings.vm"), outputFileNamePattern = "Mappings")),
    postProcessor = Some(new MarkdownPostProcessor("exchange_markdown")),
    targetFolder = "exchange_markdown_asciidocs")

  private val templates: Map[String, DocTemplateBundle] = Map(
    "mule_docs" -> muledocs,
    "markdown" -> markdown,
    "asciidoc" -> asciidoc,
    "exchange_markdown" -> exchangeMarkdown)

  def templateByName(name: String, default: DocTemplateBundle): DocTemplateBundle = {
    templates.getOrElse(name, default)
  }

  def templateByName(name: String): DocTemplateBundle = {
    templates(name)
  }

}

object WeaveDocsGenerator {
  val MODULE_DOC_PATH = "moduleDocPath"
  val MIN_DW_VERSION = "minDWVersion"

  def generate(template: DocTemplateBundle, resourceDirectory: File, outputBaseDirectory: File, configuration: java.util.Map[String, String]): Unit = {
    val asciidocFolder = new File(outputBaseDirectory, template.targetFolder)
    asciidocFolder.mkdirs()

    val dwScripts: Seq[File] = WeaveFileHelper.collectWeaveFiles(resourceDirectory, filterInternal = true)
    val basePath: String = resourceDirectory.getCanonicalPath

    //Copied all resource to root level
    template.resources.foreach(resource => {
      if (!new File(resourceDirectory, resource.name()).exists())
        Files.copy(resource.content(), new File(asciidocFolder, resource.name()).toPath)
    })

    val nameAndScripts = dwScripts.map(script => {
      val scriptPath: String = script.getCanonicalPath
      val relativePath: String = scriptPath.substring(basePath.length + 1, scriptPath.length - ".wev".length)
      val nameIdentifier: NameIdentifier = NameIdentifierHelper.fromWeaveFilePath(relativePath)
      //The idea is to parse everything at this point to segregate modules from mappings and parse them with different approaches
      val scriptNode: AstNode = WeaveFileHelper.parse(nameIdentifier, script)
      (nameIdentifier, scriptNode)
    })

    val mappings: Seq[(NameIdentifier, AstNode)] = nameAndScripts.filter(m => WeaveFileHelper.isMapping(m._2))
    val modules: Seq[(NameIdentifier, AstNode)] = {
      // We only discriminate between mappings and modules if the template has a specific renderer for mappings
      if (template.mappings.isDefined) {
        nameAndScripts.filterNot(m => WeaveFileHelper.isMapping(m._2))
      } else {
        nameAndScripts
      }
    }

    modules.foreach(nameWithNode => {
      val (nameIdentifier, weaveNode) = nameWithNode

      template.docGenerator.foreach(template => {
        val templateRunner: WeaveDocsTemplateRunner = new WeaveDocsTemplateRunner(template, asciidocFolder)
        Try(templateRunner.generateDoc(nameIdentifier, weaveNode)) match {
          case Failure(exception) =>
            throw new RuntimeException(s"Exception while executing template = ${template.docResource.name()}", exception)
          case _ =>
        }
      })
    })

    if (mappings.nonEmpty && template.mappings.isDefined) {
      val mappingsTemplateRunner: WeaveDocsTemplateRunner = new WeaveDocsTemplateRunner(template.mappings.get, asciidocFolder)
      Try(mappingsTemplateRunner.generateMappingsDoc(mappings)) match {
        case Failure(exception) =>
          throw new RuntimeException(s"Error :${exception.getMessage} while generating Mapping Docs.", exception)
        case _ =>
      }
    }

    val docTemplateIsDefined = template.tableOfContent
    if (docTemplateIsDefined.isDefined) {
      val indexTemplate: DocTemplate = docTemplateIsDefined.get
      val indexTargetFolder =
        if (indexTemplate.targetFolder.isDefined) {
          val folder = new File(asciidocFolder, indexTemplate.targetFolder.get)
          folder.mkdirs()
          folder
        } else {
          asciidocFolder
        }

      val indexTemplateRunner: WeaveDocsTemplateRunner = new WeaveDocsTemplateRunner(indexTemplate, indexTargetFolder)
      val mappingsTemplateName = if (template.mappings.isDefined) template.mappings.get.outputFileNamePattern else "mappings"
      Try(indexTemplateRunner.generateIndex(nameAndScripts, mappingsTemplateName, configuration.asScala.toMap)) match {
        case Failure(exception) =>
          throw new RuntimeException(s"Error :${exception.getMessage} while generating Index.", exception)
        case _ =>
      }
    }
    template.postProcessor.foreach(p => {
      Try(p.postExecute(outputBaseDirectory, asciidocFolder, configuration.asScala.toMap)) match {
        case Failure(exception) =>
          throw new RuntimeException(s"Error :${exception.getMessage} while executing post processor.", exception)
        case _ =>
      }
    })
  }

}

/**
  * Executed one the docs generation finished
  */
trait PostProcessor {
  /**
    * Executes a post execution
    *
    * @param outputDirectory The output directory where to generate data
    * @param templatesFolder The folder where the docs where generated
    */
  def postExecute(outputDirectory: File, templatesFolder: File, configuration: Map[String, String]): Unit

}

class HtmlPostProcessor(folder: String = "html") extends PostProcessor {

  override def postExecute(outputDirectory: File, asciidocFolder: File, configuration: Map[String, String]): Unit = {
    val asciidoctor: Asciidoctor = create()
    val htmlFolder = new File(outputDirectory, folder)
    htmlFolder.mkdirs()
    val optionsBuilder = Options.builder()
    val attributes = Attributes.builder().sourceHighlighter("coderay")
    optionsBuilder
      .toDir(htmlFolder)
      .safe(SafeMode.UNSAFE)
      .inPlace(true)
      .mkDirs(true)
      .attributes(attributes.build())
      .baseDir(asciidocFolder)
    val walker = new AsciiDocDirectoryWalker(asciidocFolder.getAbsolutePath)
    asciidoctor.convertDirectory(walker, optionsBuilder.build())

  }
}

class MarkdownPostProcessor(folder: String = "markdown") extends PostProcessor {

  override def postExecute(outputDirectory: File, asciidocFolder: File, configuration: Map[String, String]): Unit = {
    val maybeModuleDocs: Option[File] = configuration.get(WeaveDocsGenerator.MODULE_DOC_PATH).map(path => {
      new File(path)
    })
    val files = listFiles(asciidocFolder)
    if (files != null) {
      val markdownFolder = new File(outputDirectory, folder)
      markdownFolder.mkdirs()
      val hasModuleDocs: Boolean = maybeModuleDocs.isDefined && maybeModuleDocs.get.getName.endsWith("md")
      files.foreach(f => {
        val bytes = Files.readAllBytes(f.toPath)
        val markdown = AsciiDocMigrator.toMarkDown(new String(bytes, "UTF-8"))
        val markdownFile = new File(markdownFolder, FileHelper.baseName(f) + ".md")
        val writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(markdownFile), "UTF-8"))
        if (hasModuleDocs) {
          if (markdownFile.getName.matches("home.md")) {
            val moduleDocsContent = Files.readAllBytes(maybeModuleDocs.get.toPath)
            writer.write(new String(moduleDocsContent, "UTF-8"))
            writer.write(s"${System.lineSeparator()}***")
          }
        }
        writer.write(markdown)
        writer.close()
      })
    }
  }

  private def listFiles(asciidocFolder: File): Array[File] = {
    Option(asciidocFolder.listFiles()).getOrElse(Array()).flatMap(f => {
      if (f.isFile) {
        Array(f)
      } else {
        listFiles(f)
      }
    })
  }
}

case class DocTemplateBundle(docGenerator: Seq[DocTemplate], tableOfContent: Option[DocTemplate], mappings: Option[DocTemplate] = None, resources: Seq[DocResource] = Seq(), targetFolder: String = "asciidocs", postProcessor: Option[PostProcessor] = None)

trait DocResource {
  def name(): String

  def content(): InputStream

}

case class DocTemplate(docResource: DocResource, outputExtension: String = ".adoc", packageFolder: Boolean = true, outputFileNamePattern: String = "${module_name}", selector: Int = TemplateSelector.MODULE, targetFolder: Option[String] = None) {}

object TemplateSelector {
  val FUNCTIONS: Int = 0
  val FUNCTION: Int = 1
  val VARIABLES: Int = 2
  val VARIABLE: Int = 3
  val TYPES: Int = 4
  val `TYPE`: Int = 5
  val NAMESPACES: Int = 6
  val NAMESPACE: Int = 7
  val MODULE: Int = 8
  val ANNOTATIONS: Int = 9
}

class FileDocResource(file: File, outputExtension: Option[String] = None) extends DocResource {

  override def name(): String = file.getName

  override def content(): InputStream = new FileInputStream(file)

}

case class ClasspathDocResource(resourceName: String) extends DocResource {

  override def name(): String = resourceName.split("/").last

  override def content(): InputStream = getClass.getClassLoader.getResourceAsStream(resourceName)

}