package com.mulesoft.weave.docs.model

import com.mulesoft.weave.docs.FunctionDefinition
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

class RootDocModel(nameIdentifier: String, astNode: AstNode) extends DocModel(nameIdentifier, astNode) {

  def localName(): String = {
    NameIdentifier.apply(nameIdentifier).nameElements().last
  }

  def identifier(): String = nameIdentifier

  def functions: Array[FunctionDocModel] = {
    getDirectiveWith(classOf[FunctionDirectiveNode])
      .sortBy(_.variable.name)
      .filterNot(_.codeAnnotations.exists(_.name.name == "Internal")) //Ignore Internal functions
      .map(FunctionDocModel(_)).toArray
  }

  private def getDirectiveWith[T <: DirectiveNode](classType: Class[T]): Seq[T] = {
    collectDirectChildrenWith(getDirectivesContainer, classType)
  }

  def collectDirectChildrenWith[T <: AstNode](node: AstNode, classType: Class[T]): Seq[T] = {
    node.children().flatMap(child => {
      if (classType.isAssignableFrom(child.getClass)) {
        Seq(classType.cast(child))
      } else {
        Seq()
      }
    })
  }

  private def getDirectivesContainer: AstNode = {
    astNode match {
      case dn: DocumentNode => dn.header
      case _                => astNode
    }
  }

  def hasFunctions: Boolean = {
    functionsByName.nonEmpty
  }

  def functionsByName: Array[FunctionDefinition] = {
    val functionsByName: Map[String, Array[FunctionDocModel]] = functions.groupBy(function => {
      function.name
    })

    functionsByName.toSeq
      .sortBy(_._1)
      .sortBy(_._1.split("::").length)
      .map(value => {
        val shortDescription = value._2.find(fn => fn.isAnnotatedWith("GlobalDescription")) match {
          case Some(labeledFunction) =>
            labeledFunction.description()
          case None =>
            val descriptions: Array[String] = value._2.map(_.description())
            descriptions.find(_.nonEmpty).getOrElse("")
        }
        // Experimental description
        val isExperimentalFunction = value._2.exists(fn => fn.isAnnotatedWith("Experimental"))
        val description = if (isExperimentalFunction) {
          s"$shortDescription${System.lineSeparator()}${System.lineSeparator()}_Experimental:_ This function is an experimental feature that is subject to change or removal from future versions."
        } else {
          shortDescription
        }
        FunctionDefinition(value._1, value._2, description)
      })
      .toArray
  }

  def variables: Array[VariableDocModel] = {
    getDirectiveWith(classOf[VarDirective])
      .filterNot(_.codeAnnotations.exists(_.name.name == "Internal")) //Ignore Internal variables
      .sortBy(_.variable.name)
      .map(VariableDocModel(_)).toArray
  }

  def annotations: Array[AnnotationDocModel] = {
    getDirectiveWith(classOf[AnnotationDirectiveNode])
      .filterNot(_.codeAnnotations.exists(_.name.name == "Internal")) //Ignore Internal annotations
      .sortBy(_.nameIdentifier.name)
      .map(AnnotationDocModel(_)).toArray
  }

  def types: Array[TypeDocModel] = {
    getDirectiveWith(classOf[TypeDirective])
      .filterNot(_.codeAnnotations.exists(_.name.name == "Internal")) //Ignore Internal types
      .sortBy(_.variable.name)
      .map(TypeDocModel(_)).toArray
  }

  def namespaces: Array[NamespaceDocModel] = {
    getDirectiveWith(classOf[NamespaceDirective])
      .filterNot(_.codeAnnotations.exists(_.name.name == "Internal")) //Ignore Internal namespaces
      .sortBy(_.prefix.name)
      .map(NamespaceDocModel(_)).toArray
  }

  def getVersionDirective: Option[VersionDirectiveDocModel] = {
    getDirectiveWith(classOf[VersionDirective]).headOption.map(VersionDirectiveDocModel(_))
  }

  def isAnnotatedWith(annotation: String): Boolean = {
    getVersionDirective.exists(_.isAnnotatedWith(annotation))
  }

  def getSinceVersion: String = {
    getVersionDirective.map(vd => vd.getSinceVersion).getOrElse("")
  }
}

object RootDocModel {
  def apply(nameIdentifier: String, astNode: AstNode) = new RootDocModel(nameIdentifier, astNode)
}
