package com.mulesoft.weave.docs

import com.mulesoft.weave.docs.TemplateSelector._
import com.mulesoft.weave.docs.model.FunctionDocModel
import com.mulesoft.weave.docs.model.RootDocModel
import org.apache.velocity.VelocityContext
import org.apache.velocity.app.Velocity
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.io._
import scala.io.BufferedSource
import scala.io.Source
import scala.util.Try

class WeaveDocsTemplateRunner(template: DocTemplate, outputBaseFolder: File) {

  def generateDoc(nameIdentifier: NameIdentifier, weaveNode: AstNode): Unit = {
    val context: VelocityContext = new VelocityContext()
    val model = RootDocModel(nameIdentifier.name, weaveNode)
    context.put("document", model)
    context.put("module_name", nameIdentifier.nameElements().last)
    context.put("utils", new NameHelper())

    template.selector match {
      case MODULE =>
        runTemplate(nameIdentifier, context)
      case FUNCTIONS =>
        if (model.functions.nonEmpty)
          runTemplate(nameIdentifier, context)
      case VARIABLES =>
        if (model.variables.nonEmpty)
          runTemplate(nameIdentifier, context)
      case TYPES =>
        if (model.types.nonEmpty)
          runTemplate(nameIdentifier, context)
      case ANNOTATIONS =>
        if (model.annotations.nonEmpty)
          runTemplate(nameIdentifier, context)
      case NAMESPACES =>
        if (model.namespaces.nonEmpty)
          runTemplate(nameIdentifier, context)
      case VARIABLE =>
        model.variables.foreach(model => {
          context.put("variable", model)
          context.put("elementType", "Variable")
          runTemplate(nameIdentifier, context)
        })
      case FUNCTION =>
        model.functionsByName.foreach(model => {
          context.put("function", model)
          context.put("elementType", "Function")
          runTemplate(nameIdentifier, context)
        })
      case NAMESPACE =>
        model.namespaces.foreach(model => {
          context.put("namespace", model)
          context.put("elementType", "Namespace")
          runTemplate(nameIdentifier, context)
        })
      case `TYPE` =>
        model.types.foreach(model => {
          context.put("type", model)
          context.put("elementType", "Type")
          runTemplate(nameIdentifier, context)
        })
      case _ =>
    }
  }

  def generateMappingsDoc(mappingsList: Seq[(NameIdentifier, AstNode)]): Unit = {
    val mappings: Array[RootDocModel] =
      mappingsList.map(mappingElement => RootDocModel(mappingElement._1.name, mappingElement._2))
        .sortBy(_.name)
        .toArray

    val context: VelocityContext = new VelocityContext()
    context.put("mappings", mappings)
    context.put("mappingsFileName", template.outputFileNamePattern)
    val mappingsFileName: String = template.outputFileNamePattern + template.outputExtension
    val writer: Writer = new OutputStreamWriter(new FileOutputStream(new File(outputBaseFolder, mappingsFileName)), "UTF-8")

    val templateSource: BufferedSource = Source.fromInputStream(template.docResource.content(), "UTF-8")
    try {
      val tempString: String = templateSource.mkString
      Velocity.evaluate(context, writer, s"weave-docs $mappingsFileName", tempString)
    } catch {
      case e: Exception => {
        throw new RuntimeException(s"Exception while trying execute: `${template.docResource.name()}`", e)
      }
    } finally {
      Try(templateSource.close())
      writer.close()
    }

  }

  private def runTemplate(nameIdentifier: NameIdentifier, context: VelocityContext) = {
    val writer: OutputStreamWriter = getTargetFile(nameIdentifier, context)
    val source = Source.fromInputStream(template.docResource.content(), "UTF-8")
    try {
      val string = source.mkString
      Velocity.evaluate(context, writer, s"weave-docs ${nameIdentifier.name}", string)
    } finally {
      Try(source.close())
      writer.close()
    }
  }

  private def getTargetFile(nameIdentifier: NameIdentifier, context: VelocityContext): OutputStreamWriter = {
    val outputFileName = getStringTemplate(template.outputFileNamePattern, context) + template.outputExtension
    val containerFolder = if (template.packageFolder && nameIdentifier.parent().isDefined) {
      val relativeName = nameIdentifier.parent().map(_.nameElements().mkString(File.separator)).get
      val file = new File(outputBaseFolder, relativeName)
      file.mkdirs()
      file
    } else {
      outputBaseFolder
    }
    if (!containerFolder.exists()) {
      containerFolder.mkdirs()
    }
    val theFile = new File(containerFolder, outputFileName)
    if (theFile.exists()) {
      theFile.delete()
    }
    val outputStream = new FileOutputStream(theFile)
    val writer = new OutputStreamWriter(outputStream, "UTF-8")
    writer
  }

  private def getStringTemplate(template: String, context: VelocityContext): String = {
    val stringWriter = new StringWriter()
    Velocity.evaluate(context, stringWriter, "weave-docs", template)
    stringWriter.toString
  }

  def generateIndex(weaveFiles: Seq[(NameIdentifier, AstNode)], mappingsFileName: String, configuration: Map[String, String]): Unit = {
    val documents: Array[RootDocModel] = weaveFiles.map(entries => {
      RootDocModel(entries._1.name, entries._2)
    })
      .sortBy(_.name.toLowerCase)
      .toArray

    val mappings = documents.filter(docModel => WeaveFileHelper.isMapping(docModel.astNode))
    val modules = documents diff mappings

    val context: VelocityContext = new VelocityContext()
    context.put("documents", documents)
    context.put("modules", modules)
    context.put("mappings", mappings)
    context.put("mappingsFileName", mappingsFileName)
    context.put("utils", new NameHelper())
    context.put("configuration", new WeaveDocsGeneratorConfiguration(configuration))

    val tocFile: String = getStringTemplate(template.outputFileNamePattern, context) + template.outputExtension
    val writer = new OutputStreamWriter(new FileOutputStream(new File(outputBaseFolder, tocFile)), "UTF-8")
    val source = Source.fromInputStream(template.docResource.content(), "UTF-8")
    try {
      val string = source.mkString
      Velocity.evaluate(context, writer, s"weave-docs index", string)
    } finally {
      Try(source.close())
      writer.close()
    }
  }

}

class NameHelper {
  def toValidFileName(name: String): String = {
    name.toLowerCase.replace("+", "plus")
      .replace("-", "minus")
  }

  def toValidTitle(name: String): String = {
    name.replace("-", "&#45;")
      .replace("+", "&#43;")
      .replace("|", "&#124;")
      .replace(">", "&#62;")
      .replace("(", "&#40;")
      .replace(")", "&#41;")
  }
}

case class FunctionDefinition(name: String, functions: Array[FunctionDocModel], description: String)

class WeaveDocsGeneratorConfiguration(config: Map[String, String]) {

  def contains(key: String): Boolean = {
    if (config != null) {
      config.contains(key)
    } else {
      false
    }
  }

  def get(key: String): String = config.getOrElse(key, "")
}