package com.mulesoft.weave.docs.model

import com.mulesoft.weave.docs.DocsGeneratorHelper
import com.mulesoft.weave.docs.DocsGeneratorHelper.toValidContent
import com.mulesoft.weave.docs.NameHelper
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.codegen.CodeGeneratorSettings
import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.weavedoc.ExampleSectionNode
import org.mule.weave.v2.weavedoc.MoreExamplesSectionNode
import org.mule.weave.v2.weavedoc.ParameterNode
import org.mule.weave.v2.weavedoc.ParametersSectionNode
import org.mule.weave.v2.weavedoc.WeaveDocNode
import org.mule.weave.v2.weavedoc.WeaveDocParser

class FunctionDocModel(val fd: FunctionDirectiveNode) extends DocModel(fd.variable.name, fd) with AnnotationCapableDocModel {

  lazy val maybeWeaveDocNode: Option[WeaveDocNode] = {
    val weaveDoc = fd.weaveDoc
    val messageCollector = new MessageCollector
    val maybeDoc = weaveDoc.flatMap(doc => {
      WeaveDocParser.parseAst(doc.literalValue, messageCollector)
    })
    if (messageCollector.hasErrors()) {
      println("========================================================================")
      println(s"[Errors] ${messageCollector.errorMessages.size} errors while parsing documentation on: ${astNode.location().resourceName}")
      println("========================================================================")
      println(messageCollector.errorMessageString())
      println("========================================================================")
      println("")
    }
    maybeDoc
  }

  lazy val functionDescription: String = {
    val description = maybeWeaveDocNode.flatMap(_.description).map(_.literalValue).getOrElse("")
    toValidContent(description)
  }

  lazy val functionContent: String = {
    val content = maybeWeaveDocNode.flatMap(_.content).map(_.literalValue).getOrElse("")
    toValidContent(content)
  }

  def hasFunctionContent: Boolean = {
    functionContent.nonEmpty
  }

  lazy val parametersSection: String = {
    val parameters = getSectionWith(classOf[ParametersSectionNode]).headOption.map(psn => {
      if (psn.params.isEmpty) {
        ""
      } else {
        val firstParam = psn.params.head
        val params = DocsGeneratorHelper.toValidContent(psn.params.map(parameterToDocString).mkString(s"${System.lineSeparator()}"))
        val table = s"""
           |${psn.style.literalValue}
           ||===
           || Name ${if (firstParam.weaveType.isDefined) "| Type " else ""}| Description
           $params
           ||===""".stripMargin.trim
        table
      }
    }).getOrElse("")
    toValidContent(parameters)
  }

  private def parameterToDocString(parameter: ParameterNode): String = {
    val addBackTick = !parameter.name.literalValue.startsWith("`")
    s"|| ${if (addBackTick) "`" else ""}${parameter.name.literalValue}${if (addBackTick) "`" else ""} ${if (parameter.weaveType.isDefined) s"| ${parameter.weaveType.get.literalValue} " else ""}| ${parameter.description.literalValue}"
  }

  def hasParametersSection: Boolean = {
    parametersSection.nonEmpty
  }

  private def getSectionWith[T <: AstNode](classType: Class[T]): Seq[T] = {
    maybeWeaveDocNode match {
      case Some(docNode) =>
        docNode.sections.filter(s => classType.isAssignableFrom(s.getClass)).map(classType.cast(_))
      case None => Seq()
    }
  }

  lazy val examplesSections: Array[ExampleSectionNodeDocModel] = {
    getSectionWith(classOf[ExampleSectionNode]).map(e => ExampleSectionNodeDocModel(e)).toArray
  }

  lazy val moreExamplesSection: Option[MoreExamplesSectionNodeDocModel] = {
    getSectionWith(classOf[MoreExamplesSectionNode]).headOption.map(e => MoreExamplesSectionNodeDocModel(e))
  }

  def hasMoreExamplesSection: Boolean = {
    moreExamplesSection.nonEmpty
  }

  def label: String = {
    val functionNode: FunctionNode = fd.literal.asInstanceOf[FunctionNode]
    val writer = new StringCodeWriter()
    val generator = new CodeGenerator(writer, CodeGeneratorSettings())
    generator.generate(fd.variable)
    if (functionNode.typeParameterList.isDefined) {
      writer
        .print("<")
        .printForeachWithSeparator(", ", functionNode.typeParameterList.get.children(), (node: AstNode) => generator.generate(node))
        .print(">")
    }
    generator.generate(functionNode.params)
    generator.generateTypeDeclaration(functionNode.returnType)
    writer.toString()
  }

  override def annotationCapableNode: AnnotationCapableNode = fd

  def getReplacementFunction: String = {
    getReplacementArgument.getOrElse("")
  }

  def getReplacementArgument: Option[String] = {
    codeAnnotation("Deprecated").map(an => AnnotationNodeHelper.argString("replacement", an).getOrElse(""))
  }

  def getReplacementLink: String = {
    getReplacementArgument.map(replacement => {
      val split = replacement.split("::")
      val module = if (split.nonEmpty) {
        split.take(split.length - 1).mkString("-").toLowerCase()
      } else {
        ""
      }
      val functionName = new NameHelper().toValidFileName(split.last.toLowerCase())
      s"$module-functions-$functionName.adoc"
    }).getOrElse("")
  }
}

object FunctionDocModel {
  def apply(node: FunctionDirectiveNode): FunctionDocModel = new FunctionDocModel(node)
}