package org.mule.weave.maven.plugin;

import static java.lang.String.format;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.codehaus.plexus.util.FileUtils;
import org.mule.weave.v2.inspector.NoInspector$;
import org.mule.weave.v2.parser.DocumentParser;
import org.mule.weave.v2.parser.Message;
import org.mule.weave.v2.parser.ast.AstNode;
import org.mule.weave.v2.parser.ast.variables.NameIdentifier;
import org.mule.weave.v2.parser.location.WeaveLocation;
import org.mule.weave.v2.parser.phase.ParsingContext;
import org.mule.weave.v2.parser.phase.ParsingResult;
import org.mule.weave.v2.parser.phase.PhaseResult;
import org.mule.weave.v2.sdk.ParsingContextFactory;
import org.mule.weave.v2.sdk.WeaveResourceFactory;
import scala.Option;
import scala.Tuple2;
import scala.collection.Iterator;

import java.io.File;
import java.io.IOException;
import java.util.Optional;

@Mojo(name = "compile",
        defaultPhase = LifecyclePhase.COMPILE,
        requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
        requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME,
        executionStrategy = "always")
@Execute(goal = "compile")
public class WeaveCompileMojo extends AbstractWeaveMojo {
    
    private static final String DW_EXTENSION = "dwl";
    
    @Override
    public void execute() throws MojoFailureException {
        getLog().info(format("Start Compiling DataWeave: %s" , project.getName()));
        final int numberOfError = validate(sourceFolder, "");
        if (numberOfError > 0) {
            throw new MojoFailureException(format("Compilation failure %s errors found.", numberOfError));
        }
        getLog().info(format("DataWeave %s successfully compiled", project.getName()));
    }

    private int validate(File sourceDir, String name) {
        int errors = 0;
        final File[] child = Optional.ofNullable(sourceDir.listFiles()).orElse(new File[0]);
        for (File file : child) {
            getLog().info(format("Compiling %s", file.getAbsolutePath()));
            if (isDWFile(file)) {
                final String basename = FileUtils.basename(file.getName());
                errors += parse(NameIdentifier.apply(nameIdentifier(name, basename), Option.empty()), file);
            } else if (file.isDirectory()) {
                validate(file, nameIdentifier(name, file.getName()));
            }
        }
        return errors;
    }
    
    private boolean isDWFile(File file) {
        return Optional.ofNullable(file).map(f -> DW_EXTENSION.equals(FileUtils.extension(f.getName()))).orElse(false);
    }

    private String nameIdentifier(String name, String moduleName) {
        if (name.isEmpty()) {
            return moduleName;
        }
        return name + NameIdentifier.SEPARATOR() + moduleName;
    }

    private int parse(NameIdentifier identifier, File file) {
        final DocumentParser documentParser = new DocumentParser(15, NoInspector$.MODULE$);
        final ParsingContext parsingContext = ParsingContextFactory.createParsingContext(identifier);
        final PhaseResult<ParsingResult<AstNode>> parseResult = documentParser.parse(WeaveResourceFactory.fromFile(file), parsingContext);
        final Iterator<Tuple2<WeaveLocation, Message>> errors = parseResult.messages().errorMessages().toIterator();
        while (errors.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = errors.next();
            final WeaveLocation weaveLocation = next._1;
            getLog().error(format("%s at %s ", next._2.message(), weaveLocation.locationString()));
            try {
                getLog().error(format("%s: [%s, %s]", file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column()));
            } catch (IOException ignored) {
                // Nothing to do
            }
        }

        final Iterator<Tuple2<WeaveLocation, Message>> warnings = parseResult.messages().warningMessages().toIterator();
        while (warnings.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = warnings.next();
            final WeaveLocation weaveLocation = next._1;
            getLog().warn(format("%s at %s", next._2.message(), weaveLocation.locationString()));
            try {
                getLog().warn(format("%s: [%s, %s]", file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column()));
            } catch (IOException ignored) {
                // Nothing to do
            }
        }
        
        return parseResult.errorMessages().length();
    }
}
