package org.mule.weave.maven.plugin;

import static java.lang.Boolean.getBoolean;
import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static java.util.regex.Pattern.compile;
import static org.mule.weave.maven.plugin.Constants.SKIP_DEPLOY_DOCS;

import org.apache.maven.model.DeploymentRepository;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.crypto.DefaultSettingsDecryptionRequest;
import org.apache.maven.settings.crypto.SettingsDecrypter;
import org.apache.maven.settings.crypto.SettingsDecryptionResult;
import org.mule.weave.maven.plugin.exchange.client.ExchangeClient;
import org.mule.weave.maven.plugin.exchange.client.MavenExchangeClientLogger;

import java.io.File;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Mojo(name = "deploy-docs",
        defaultPhase = LifecyclePhase.DEPLOY
)
@Execute(goal = "deploy-docs")
public class WeaveDeployDocsMojo extends WeaveDocsMojo {
    
    private static final Pattern exchangeAnypointDomain = compile("https://maven\\.(?<baseUri>.*anypoint\\.mulesoft\\.com)");
    
    private static final String MARKDOWN_FOLDER ="exchange_markdown";

    @Component
    protected SettingsDecrypter decrypter;

    @Override
    public void doExecute() {
        if (getBoolean(SKIP_DEPLOY_DOCS)) {
            getLog().warn("Skipping Deploy of the WeaveDocs.");
            return;
        }
        final DeploymentRepository repository;
        try {
            repository = getDeploymentRepository();

        } catch (MojoFailureException e) {
            getLog().warn(format("Unable to get deployment repository: %s", e.getMessage()));
            return;
        }
        final String url = repository.getUrl();
        final Matcher matcher = exchangeAnypointDomain.matcher(url);
        final Optional<File> maybeFavicon = ofNullable(config.getFavicon());
        if (matcher.find()) {
            final String anypointUrl = "https://" + matcher.group("baseUri");
            final Settings settings = loadDefaultUserSettings();
            if (settings != null) {
                final Server server = settings.getServer(repository.getId());
                if (server != null) {
                    File exchangeOutputDoc = new File(config.getOutput(), MARKDOWN_FOLDER);
                    if (exchangeOutputDoc.exists()) {
                        final Server decryptedServer = decrypt(server);
                        final String username = decryptedServer.getUsername();
                        final String password = decryptedServer.getPassword();
                        
                        getLog().info(format("Deploying: `%s` to : `%s`", exchangeOutputDoc.getAbsolutePath(), anypointUrl));
                                
                        ExchangeClient client = new ExchangeClient.Builder()
                                .withBaseAnypointUrl(anypointUrl)
                                .withUsername(username)
                                .withPassword(password)
                                .withLogger(new MavenExchangeClientLogger(getLog()))
                                .build();
                        
                        client.uploadPortalPages(exchangeOutputDoc, project, maybeFavicon);
                        
                        getLog().info(format("Deployed successful to: `%s`", anypointUrl));
                    } else {
                        getLog().warn(format("Ignoring deploy docs. Docs folder: %s was not found", exchangeOutputDoc.getAbsolutePath()));
                    }
                } else {
                    getLog().warn("Ignoring deploy docs as server was not found");
                }
            }
        } else {
            getLog().warn(format("Unable to find anypoint url from: %s", url));
        }
    }

    private Settings loadDefaultUserSettings() {
        return session.getSettings();
    }

    private Server decrypt(Server server) {
        final DefaultSettingsDecryptionRequest decryptionRequest = new DefaultSettingsDecryptionRequest(server);
        SettingsDecryptionResult decryptedResult = decrypter.decrypt(decryptionRequest);
        return decryptedResult.getServer();
    }
}