package org.mule.weave.maven.plugin;

import static java.lang.String.format;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.cli.CommandLineException;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.codehaus.plexus.util.cli.Commandline;
import org.codehaus.plexus.util.cli.StreamConsumer;

import java.util.Map;

public class WeaveTestCommandlineExecutor {

    private final MavenProject project;
    private final Log log;
    private final String classpath;
    private final String[] args;
    private final WeaveTestMojoConfig mojoConfig;


    public WeaveTestCommandlineExecutor(MavenProject project, Log log, String classpath, String[] args, WeaveTestMojoConfig mojoConfig) {
        this.project = project;
        this.log = log;
        this.classpath = classpath;
        this.args = args;
        this.mojoConfig = mojoConfig;
    }

    // Returns true if all tests pass
    public boolean execute() throws MojoFailureException {
        final Commandline cli = new Commandline();
        cli.setWorkingDirectory(project.getBasedir());
        cli.setExecutable("java");

        // Set up environment
        if (mojoConfig.getRunnerEnvironmentVariables() != null) {
            for (final Map.Entry<String, String> entry : mojoConfig.getRunnerEnvironmentVariables().entrySet()) {
                cli.addEnvironment(entry.getKey(), entry.getValue());
            }
        }
        cli.addEnvironment("CLASSPATH", classpath);

        // Set up system properties
        if (mojoConfig.getRunnerSystemProperties() != null) {
            for (final Map.Entry<String, String> entry : mojoConfig.getRunnerSystemProperties().entrySet()) {
                cli.createArg().setValue(format("-D%s=%s", entry.getKey(), entry.getValue()));
            }
        }

        if (mojoConfig.isRunnerJvmDebug()) {
            cli.createArg().setValue("-agentlib:jdwp=transport=dt_socket,server=y,suspend=y,address=5005");
        }

        cli.createArg().setValue(format("-Dbasedir=%s", project.getBasedir().getAbsolutePath()));

        // Set user specified JVM arguments
        if (mojoConfig.getRunnerArgLine() != null) {
            cli.createArg().setLine(mojoConfig.getRunnerArgLine());
        }

        // Set ScalaTest arguments
        cli.createArg().setValue("org.mule.weave.v2.module.test.runner.TestRunner");
        
        if (args != null) {
            for (final String arg : args) {
                cli.createArg().setValue(arg);
            }    
        }
        // Log command string
        final String commandLogStatement = "Running DW Command: " + cli;
        if (mojoConfig.isRunnerLogForkedProcessCommand()) {
            log.info(commandLogStatement);
        } else {
            log.debug(commandLogStatement);
        }
        
        try {
            final int result = executeCommandLine(cli);
            return result == 0;
        } catch (final CommandLineException e) {
            throw new MojoFailureException("Exception while executing forked process.", e);
        }
    }

    public MavenProject getProject() {
        return project;
    }

    public Log getLog() {
        return log;
    }

    public String getClasspath() {
        return classpath;
    }

    public String[] getArgs() {
        return args;
    }

    public WeaveTestMojoConfig getMojoConfig() {
        return mojoConfig;
    }

    protected int executeCommandLine(Commandline cli) throws CommandLineException {
        final StreamConsumer streamConsumer = System.out::println;
        return CommandLineUtils.executeCommandLine(cli, streamConsumer, streamConsumer);
    }
}
