package org.mule.weave.maven.plugin;

import org.apache.maven.archiver.MavenArchiveConfiguration;
import org.apache.maven.archiver.MavenArchiver;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.DefaultArtifact;
import org.apache.maven.artifact.handler.ArtifactHandler;
import org.apache.maven.artifact.handler.manager.ArtifactHandlerManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.shared.model.fileset.FileSet;
import org.codehaus.plexus.archiver.Archiver;
import org.codehaus.plexus.archiver.jar.JarArchiver;

import java.io.File;
import java.util.Arrays;
import java.util.Map;

@Mojo(name = "package",
        defaultPhase = LifecyclePhase.PACKAGE,
        requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
        requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
@Execute(goal = "package")
public class WeavePackagerMojo extends AbstractWeaveMojo {
    
    public static final String ARTIFACT_TYPE = "jar";
    public static final String ARTIFACT_CLASSIFIER = "dw-library";
    
    /**
     * Directory containing the generated JAR.
     */
    @Parameter(defaultValue = "${project.build.directory}", required = true, readonly = true)
    protected File outputDirectory;

    /**
     * Name of the generated JAR.
     */
    @Parameter(defaultValue = "${project.build.finalName}", readonly = true)
    protected String finalName;

    /**
     * Directory containing the classes and resource files that should be packaged into the JAR.
     */
    @Parameter(defaultValue = "${project.build.outputDirectory}", required = true)
    protected File classesDirectory;

    /**
     * The Jar archiver.
     */
    @Component
    protected Map<String, Archiver> archivers;

    /**
     * The archive configuration to use. See <a href="http://maven.apache.org/shared/maven-archiver/index.html">Maven
     * Archiver Reference</a>.
     */
    private final MavenArchiveConfiguration archive = new MavenArchiveConfiguration();

    @Component
    protected ArtifactHandlerManager artifactHandlerManager;

    /**
     * Generates the JAR.
     *
     * @throws MojoExecutionException in case of an error.
     */
    public void execute() throws MojoExecutionException {
        final File jarFile = createArchive();
        final ArtifactHandler handler = artifactHandlerManager.getArtifactHandler(ARTIFACT_TYPE);
        Artifact artifact = new DefaultArtifact(project.getGroupId(), project.getArtifactId(), project.getVersion(), null, ARTIFACT_TYPE, ARTIFACT_CLASSIFIER, handler);
        artifact.setFile(jarFile);
        artifact.setResolved(true);
        project.setArtifact(artifact);
    }
    
    /**
     * Returns the Jar file to generate, based on an optional classifier.
     *
     * @param basedir         the output directory
     * @param resultFinalName the name of the ear file
     * @param classifier      an optional classifier
     * @return the file to generate
     */
    private File getJarFile(File basedir, String resultFinalName, String classifier) {
        if (basedir == null) {
            throw new IllegalArgumentException("basedir is not allowed to be null");
        }
        if (resultFinalName == null) {
            throw new IllegalArgumentException("finalName is not allowed to be null");
        }

        String fileName = resultFinalName + "-" + classifier + ".jar";
        return new File(basedir, fileName);
    }

    /**
     * Generates the JAR.
     *
     * @return The instance of File for the created archive file.
     * @throws MojoExecutionException in case of an error.
     */
    private File createArchive() throws MojoExecutionException {
        final String[] includes = {"**/**"};
        final String[] excludes = {"**/package.html"};
        File jarFile = getJarFile(outputDirectory, finalName, ARTIFACT_CLASSIFIER);
        FileSet jarContentFileSet = new FileSet();
        jarContentFileSet.setDirectory(classesDirectory.getAbsolutePath());
        jarContentFileSet.setIncludes(Arrays.asList(includes));
        jarContentFileSet.setExcludes(Arrays.asList(excludes));

        MavenArchiver archiver = new MavenArchiver();
        archiver.setArchiver((JarArchiver) archivers.get(ARTIFACT_TYPE));
        archiver.setOutputFile(jarFile);
        archive.setForced(true);
        try {
            File contentDirectory = classesDirectory;
            if (!contentDirectory.exists()) {
                getLog().warn("JAR will be empty - no content was marked for inclusion!");
            } else {
                archiver.getArchiver().addDirectory(contentDirectory, includes, excludes);
            }
            archiver.createArchive(session, project, archive);
            return jarFile;
        } catch (Exception e) {
            // TODO: improve error handling
            throw new MojoExecutionException("Error creating weave ", e);
        }
    }
}
