package org.mule.weave.maven.plugin;

import static java.lang.System.getProperty;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.mule.weave.maven.plugin.Constants.SKIP_TESTS;
import static org.mule.weave.maven.plugin.MojoUtils.compoundArg;
import static org.mule.weave.maven.plugin.MojoUtils.concat;

import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.codehaus.plexus.util.cli.Commandline;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

@Mojo(name = "test",
        defaultPhase = LifecyclePhase.TEST,
        requiresDependencyCollection = ResolutionScope.TEST,
        requiresDependencyResolution = ResolutionScope.TEST)
@Execute(goal = "test")
public class WeaveTestMojo extends AbstractWeaveMojo {

    @Parameter(name = "tests")
    protected WeaveTestMojoConfig config;

    private CommandLineRunner commandLineRunner;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (getProperty(SKIP_TESTS) != null) {
            getLog().warn("Skipping Weave Tests.");
            return;
        }
        try {
            init();
            
            List<String> dwITArgs = new ArrayList<>();
            String classpath = buildClassPathEnvironment();
            String[] args = concat(dwITArgs, testsDirectories(), reports(), coverage(), aggregators());
            Commandline commandline = new WeaveTestCommandlineRequestBuilder(project, classpath, args, config).build();
            // Log command string
            final String commandLogStatement = "Running DW Command: " + commandline;
            if (config.isRunnerLogForkedProcessCommand()) {
                getLog().info(commandLogStatement);
            } else {
                getLog().debug(commandLogStatement);
            }

            final int result = commandLineRunner.executeCommandLine(commandline);
            boolean success = result == 0;
            if (!success) {
                throw new MojoFailureException("Tests Failed");
            }
        } catch (Exception e) {
            throw new MojoExecutionException("Error", e);
        }
    }
    
    private void init() throws IOException {
        if (config == null) {
            config = new WeaveTestMojoConfig(getOutputFolder());
        }
        
        if (config.getOutput() == null) {
            config.setOutput(getOutputFolder());
        }

        if (dwTestFolder.getCanonicalFile().exists()) {
            if (config.getRunnerSystemProperties() == null) {
                config.setRunnerSystemProperties(new HashMap<>());
            }
            config.getRunnerSystemProperties().put("dwtestDir", dwTestFolder.getCanonicalPath());
            config.getRunnerSystemProperties().put("dwtestResources", dwTestResources.getCanonicalPath());
        }

        if (commandLineRunner == null) {
            commandLineRunner = new DefaultCommandLineRunner();
        }
    }
    
    private File getOutputFolder() {
       return new File(project.getBuild().getDirectory() + File.separator + "data-weave-test-reports");
    }

    private String buildClassPathEnvironment() {
        StringBuffer buffer = new StringBuffer();
        boolean first = true;
        for (String e : testClasspathElements()) {
            if (first) {
                first = false;
            } else {
                buffer.append(File.pathSeparator);
            }
            buffer.append(e);
        }
        return buffer.toString();
    }

    // Have to use the programmatic way of getting the classpath elements
    // instead of the field-level injection since that apparently doesn't work
    // for ReporterMojos in maven-2.2 (it does work in maven-3)
    private List<String> testClasspathElements() {
        try {
            return project.getTestClasspathElements();
        } catch (DependencyResolutionRequiredException e) {
            // There's really no known way this exception can happen since
            // the @requiresDependencyResolution at the top of the class
            // defines test-scoped resolution.
            throw new IllegalStateException("Dependency resolution should be test-scoped.", e);
        }
    }

    private List<String> testsDirectories() throws IOException {
        return compoundArg("-R", dwTestFolder.getCanonicalPath());
    }

    private List<String> reports() throws IOException {
        return asList("-r", 
                "junit=" + config.getOutput().getCanonicalPath(),
                "-r",
                "json=" + config.getOutput().getCanonicalPath()
        );
    }

    private List<String> coverage()  throws IOException {
        if (config.isCoverageEnabled()) {
            return asList(
                    "-coverage",
                    "src/main/dw",
                    "-coverageReport",
                    config.getCoverageFormat() + "=" + config.getOutput().getCanonicalPath() + "/coverage");
        } else {
            return emptyList();
        }
    }

    private List<String> aggregators() throws IOException {
        return config.isHtmlReport() ? asList("-a", "html=" + config.getOutput().getCanonicalPath()) : emptyList();
    }

    public void setTests(WeaveTestMojoConfig config) {
        this.config = config;
    }

    public void setCommandLineRunner(CommandLineRunner commandLineRunner) {
        this.commandLineRunner = commandLineRunner;
    }
}
