package org.mule.weave.maven.plugin;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.*;
import org.codehaus.plexus.util.cli.Commandline;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static java.lang.System.getProperty;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.mule.weave.maven.plugin.Constants.SKIP_TESTS;
import static org.mule.weave.maven.plugin.MojoUtils.compoundArg;
import static org.mule.weave.maven.plugin.MojoUtils.concat;

@Mojo(name = "test",
        defaultPhase = LifecyclePhase.TEST,
        requiresDependencyCollection = ResolutionScope.TEST,
        requiresDependencyResolution = ResolutionScope.TEST)
@Execute(goal = "test")
public class WeaveTestMojo extends AbstractWeaveMojo {

    @Parameter(name = "tests")
    protected WeaveTestMojoConfig config;

    private CommandLineRunner commandLineRunner;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (getProperty(SKIP_TESTS) != null) {
            getLog().warn("Skipping Weave Tests.");
            return;
        }
        try {
            init();
            
            List<String> dwITArgs = new ArrayList<>();
            String classpath = buildTestClassPathEnvironment();
            String[] args = concat(dwITArgs, testsDirectories(), reports(), coverage(), aggregators());
            Commandline commandline = new WeaveTestCommandlineRequestBuilder(project, classpath, args, config).build();
            // Log command string
            final String commandLogStatement = "Running DW Command: " + commandline;
            if (config.isRunnerLogForkedProcessCommand()) {
                getLog().info(commandLogStatement);
            } else {
                getLog().debug(commandLogStatement);
            }

            final int result = commandLineRunner.executeCommandLine(commandline);
            boolean success = result == 0;
            if (!success) {
                throw new MojoFailureException("Tests Failed");
            }
        } catch (Exception e) {
            throw new MojoExecutionException("Error", e);
        }
    }

    private void init() throws IOException {
        if (config == null) {
            config = new WeaveTestMojoConfig(getOutputFolder());
        }

        if (config.getOutput() == null) {
            config.setOutput(getOutputFolder());
        }

        if (dwTestFolder.getCanonicalFile().exists()) {
            if (config.getRunnerSystemProperties() == null) {
                config.setRunnerSystemProperties(new HashMap<>());
            }
            config.getRunnerSystemProperties().put("dwtestDir", dwTestFolder.getCanonicalPath());
            config.getRunnerSystemProperties().put("dwtestResources", dwTestResources.getCanonicalPath());
        }

        if (commandLineRunner == null) {
            commandLineRunner = new DefaultCommandLineRunner();
        }
    }

    private File getOutputFolder() {
        return new File(project.getBuild().getDirectory() + File.separator + "data-weave-test-reports");
    }

    private List<String> testsDirectories() throws IOException {
        return compoundArg("-R", dwTestFolder.getCanonicalPath());
    }

    private List<String> reports() throws IOException {
        return asList("-r",
                "junit=" + config.getOutput().getCanonicalPath(),
                "-r",
                "json=" + config.getOutput().getCanonicalPath()
        );
    }

    private List<String> coverage() throws IOException {
        if (config.isCoverageEnabled()) {
            return asList(
                    "-coverage",
                    sourceFolder.getPath(),
                    "-coverageReport",
                    config.getCoverageFormat() + "=" + config.getOutput().getCanonicalPath() + "/coverage");
        } else {
            return emptyList();
        }
    }

    private List<String> aggregators() throws IOException {
        return config.isHtmlReport() ? asList("-a", "html=" + config.getOutput().getCanonicalPath()) : emptyList();
    }

    public void setTests(WeaveTestMojoConfig config) {
        this.config = config;
    }

    public void setCommandLineRunner(CommandLineRunner commandLineRunner) {
        this.commandLineRunner = commandLineRunner;
    }
}
