package org.mule.weave.maven.plugin;

import static java.lang.String.format;

import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import java.io.File;
import java.util.List;

/**
 * Add more source directories to the POM.
 */
@Mojo(name = "initialize", 
        executionStrategy = "always", 
        defaultPhase = LifecyclePhase.INITIALIZE, 
        threadSafe = true)
@Execute(goal = "initialize")
public class WeaveProjectInitializeMojo extends AbstractWeaveMojo {
    
    /**
     * Should use CanonicalPath to normalize path (true =&gt; getCanonicalPath, false =&gt;
     * getAbsolutePath)
     *
     * @see <a href="https://github.com/davidB/maven-scala-plugin/issues/50">#50</a>
     */
    @Parameter(property = "maven.scala.useCanonicalPath", defaultValue = "true")
    private boolean useCanonicalPath;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        try {
            getLog().debug("Initializing DataWeave Maven Plugin...");
            if (sourceFolder != null && sourceFolder.exists()) {
                final String path = pathOf(sourceFolder, useCanonicalPath);
                if (resourceNotDefined(path, project.getResources())) {
                    getLog().debug(format("Add Source directory: %s", path));
                    final Resource resource = new Resource();
                    resource.setDirectory(path);
                    project.addResource(resource);
                }
            }
            if (dwTestFolder != null && dwTestFolder.exists()) {
                final String path = pathOf(dwTestFolder, useCanonicalPath);
                defineTestResourceIfMissing(path);
            }
        } catch (final Exception exc) {
            throw new MojoExecutionException("Error while running initialize.", exc);
        }
    }

    private void defineTestResourceIfMissing(String path) {
        final List<Resource> testResources = project.getTestResources();
        if (resourceNotDefined(path, testResources)) {
            getLog().debug(format("Add Test Source directory: %s", path));
            final Resource testResource = new Resource();
            testResource.setDirectory(path);
            project.addTestResource(testResource);
        }
    }

    private boolean resourceNotDefined(String path, List<Resource> testResources) {
        return testResources.stream().noneMatch((r) -> r.getDirectory().equals(path));
    }

    /**
     * @param canonical Should use CanonicalPath to normalize path (true => getCanonicalPath, false
     *                  =&gt; getAbsolutePath)
     * @see <a href="https://github.com/davidB/maven-scala-plugin/issues/50">#50</a>
     */
    static String pathOf(File f, boolean canonical) throws Exception {
        return canonical ? f.getCanonicalPath() : f.getAbsolutePath();
    }
}

