package org.mule.weave.maven.plugin.exchange.client;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.nio.file.Files.readAllBytes;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.client.Entity.entity;
import static org.apache.commons.io.FilenameUtils.getBaseName;
import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.project.MavenProject;
import org.mule.weave.maven.plugin.client.RestClient;
import org.mule.weave.maven.plugin.exchange.client.model.AssetIdBody;
import org.mule.weave.maven.plugin.exchange.client.model.AuthorizationResponse;
import org.mule.weave.maven.plugin.exchange.client.model.Credentials;
import org.mule.weave.maven.plugin.exchange.client.model.OAuthTokenResponse;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ExchangeClient extends RestClient {
    public static final String PROD_ENV = "https://anypoint.mulesoft.com";
    private static final String LOGIN = "/accounts/login";
    private static final String OAUTH_LOGIN = "accounts/api/v2/oauth2/token";
    private static final String CLIENT_CA_PLACEHOLDER = "~~~Client~~~";
    private static final String CREDS_SEPARATOR = "~?~";
    private static final String EXCHANGE_URL = "/exchange/api/v2";
    private static final String ASSET_URL = EXCHANGE_URL + "/assets/%s/%s";
    private static final String PORTAL_URL = ASSET_URL + "/%s/portal";
    private static final String DRAFT_PAGES_URL = PORTAL_URL + "/draft/pages/%s";
    private static final String ICON_URL = ASSET_URL + "/icon";
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final List<String> ICON_VALID_EXTENSIONS =  Arrays.asList("jpg", "png", "svg");
    private static final String MAPPINGS_FILE_NAME = "Mappings.md";
    private static final String TEXT_MARKDOWN = "text/markdown";
    private static final String IMAGE_PNG = "image/png";

    // Exchange Invalid Characters matching regex
    private static final String EXCHANGE_INVALID_CHARS = "[\\@|\\*|\\/|\\\\|\\/|\\_|\\+]";
    private static final Pattern EXCHANGE_INVALID_CHARS_PATTERN = Pattern.compile(EXCHANGE_INVALID_CHARS);
    
    private final String baseAnypointUrl;
    private final String username;
    private final String password;
    private final ExchangeClientLogger logger;
    private String bearerToken;
    
    public ExchangeClient(String baseAnypointUrl, String username, String password, ExchangeClientLogger logger) {
        super(true, true);
        this.baseAnypointUrl = baseAnypointUrl;
        this.username = username;
        this.password = password;
        this.logger = logger;
    }
    
    public String getBearerToken() {
        if (isBlank(bearerToken)) {
            if(username.equals(CLIENT_CA_PLACEHOLDER)){
                String clientId = StringUtils.substringBefore(password, CREDS_SEPARATOR);
                String clientSecret = StringUtils.substringAfter(password, CREDS_SEPARATOR);
                Form form = new Form()
                        .param("client_id", clientId)
                        .param("client_secret", clientSecret)
                        .param("grant_type", "client_credentials");
                OAuthTokenResponse response = postFormUrlEncoded(baseAnypointUrl, OAUTH_LOGIN, Entity.form(form), OAuthTokenResponse.class);
                bearerToken = response.getAccessToken();
            } else {
                Credentials credentials = new Credentials();
                credentials.setUsername(username);
                credentials.setPassword(password);
                AuthorizationResponse response = post(baseAnypointUrl, LOGIN, credentials, AuthorizationResponse.class);
                bearerToken = response.getAccessToken();
            }
        }
        return bearerToken;
    }
    
    public void uploadPortalPages(File docsFolder, MavenProject project, Optional<File> maybeFavicon) {
        logger.info("Publishing portal pages ...");
        
        String groupId = project.getGroupId();
        String artifactId = project.getArtifactId();
        String version = project.getVersion();


        uploadDrafts(docsFolder, groupId, artifactId, version);
        
        updateAsset(groupId, artifactId, project.getName(), project.getDescription());
        
        updateFavicon(groupId, artifactId, maybeFavicon);
        
        publishPortal(groupId, artifactId, version);
        
        logger.info("Published portal pages successfully.");
    }
    
    public void publishPortal(String groupId, String artifactId, String version) {
        try {
            String path = format(PORTAL_URL, encodeUtf8(groupId), encodeUtf8(artifactId), encodeUtf8(version));
            patch(baseAnypointUrl, path, null, this::addAuthorizationHeader);
        } catch (Exception e) {
            logger.error("An error has occurred publishing portal.", e);
            throw new RuntimeException(e);
        }
    }
    
    private void addAuthorizationHeader(Invocation.Builder builder) {
        builder.header(AUTHORIZATION_HEADER, "bearer " + getBearerToken());
    }

    public void uploadDrafts(File docsFolder, String groupId, String artifactId, String version) {
        logger.info("Publishing drafts pages from `" + docsFolder.getName() + "`...");
        File[] files = ofNullable(docsFolder.listFiles()).orElse(new File[]{});
        List<File> sortedFiles = Arrays.stream(files)
                .filter(file -> !MAPPINGS_FILE_NAME.equals(file.getName()))
                .sorted()
                .collect(toList());
        
        Optional<File> maybeMappings = Arrays.stream(files).filter(file -> MAPPINGS_FILE_NAME.equals(file.getName())).findFirst();
        maybeMappings.ifPresent(sortedFiles::add);
 
        boolean fileNamesContainSpecialChars = sortedFiles.stream().anyMatch(file -> {
            Matcher matcher = EXCHANGE_INVALID_CHARS_PATTERN.matcher(file.getName());
            return matcher.find();
        });

        if (fileNamesContainSpecialChars && !sortedFiles.isEmpty()) {
            logger.error("File names cannot contain these characters: @ * + / _ \\\\");
            throw new RuntimeException("File names cannot contain these characters: @ * + / _ \\\\");
        } else {
            sortedFiles.forEach(file -> {
                if (file.isDirectory()) {
                    uploadDrafts(file, groupId, artifactId, version);
                } else if ("md".equals(getExtension(file.getName()))){
                    String pagePath = getBaseName(file.getName());
                    logger.info("Publishing draft page `"+ pagePath + "`....");
                    try {
                        Path filePath = Paths.get(file.getAbsolutePath());
                        byte[] bytes = readAllBytes(filePath);
                        String content = new String(bytes, StandardCharsets.UTF_8);
                        String path = format(DRAFT_PAGES_URL, groupId, artifactId, version, pagePath);
                        put(baseAnypointUrl, path, entity(content, TEXT_MARKDOWN), this::addAuthorizationHeader);
                        logger.info("Draft page `" + pagePath + "` published successfully.");
                    } catch (Exception e) {
                        logger.error(format("Error while publishing draft page `%s`.",  pagePath), e);
                        throw new RuntimeException(e);
                    }
                }
            });
        }
        logger.info("Draft pages from `" + docsFolder.getName() + "` published successfully.");
    }

    public void updateAsset(String groupId, String artifactId, String name, String description) {
        logger.info("Updating asset ...");
        if (isNotBlank(name) || isNotBlank(description)) {
            try {
                AssetIdBody body = new AssetIdBody();
                if (isNotBlank(name)) {
                    body.setName(name);
                }
                if (isNotBlank(description)) {
                    body.setDescription(description);
                }
                String path = format(ASSET_URL, encodeUtf8(groupId), encodeUtf8(artifactId));
                patch(baseAnypointUrl, path, body, this::addAuthorizationHeader);
                logger.info("Asset updated successfully.");
            } catch (Exception e) {
                logger.error("Error while updating the asset.", e);
                throw new RuntimeException(e);
            }
        } else {
            logger.warn("Project name and description are not defined in the project's pom.xml. Name and description update will not be executed.");
        }
    }

    public void updateFavicon(String groupId, String artifactId, Optional<File> maybeFavicon) {
        logger.info("Updating icon ...");
        if (maybeFavicon.isPresent()) {
            File favicon = maybeFavicon.get();
            if (favicon.exists()) {
                String iconExtension = getExtension(favicon.getName());
                if (ICON_VALID_EXTENSIONS.contains(iconExtension)) {
                    try {
                        Path faviconPath = Paths.get(favicon.getAbsolutePath());
                        byte[] bytes = readAllBytes(faviconPath);
                        String path = format(ICON_URL, encodeUtf8(groupId), encodeUtf8(artifactId));
                        put(baseAnypointUrl, path, entity(bytes, IMAGE_PNG), this::addAuthorizationHeader);
                        logger.info("Icon updated successfully.");
                    } catch (Exception e) {
                        logger.error("An error has occurred attempting to update the asset's icon.", e);
                        throw new RuntimeException(e);
                    }
                } else {
                    logger.warn(format("Invalid icon extension: extension: %s, valid: %s. Icon update will not be executed.", iconExtension, join(",", ICON_VALID_EXTENSIONS)));
                }
            } else {
                logger.warn("Invalid icon file configured. Icon update will not be executed.");
            }
        } else {
            logger.warn("Property faviconPath was not defined in the project's pom.xml. Icon update will not be executed.");
        }
    }
    
    public static class Builder {
        private String baseAnypointUrl;
        private String username;
        private String password;
        private MavenExchangeClientLogger logger;
        
        public Builder withBaseAnypointUrl(String baseAnypointUrl) {
            this.baseAnypointUrl = baseAnypointUrl;
            return this;
        }
        
        public Builder withUsername(String username) {
            this.username = username;
            return this;
        }

        public Builder withPassword(String password) {
            this.password = password;
            return this;
        }

        public Builder withLogger(MavenExchangeClientLogger logger) {
            this.logger = logger;
            return this;
        }
        
        public ExchangeClient build() {
            if (isBlank(baseAnypointUrl)) {
                baseAnypointUrl = PROD_ENV;
            }
            return new ExchangeClient(baseAnypointUrl, username, password, logger);
        }
    }
}


