package org.mule.weave.maven.plugin;

import com.mulesoft.weave.compiler.WeaveBinaryCompiler;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.*;
import org.mule.weave.v2.api.tooling.message.ValidationMessage;
import org.mule.weave.v2.inspector.NoInspector$;
import org.mule.weave.v2.module.pojo.ClassLoaderServiceAware;
import org.mule.weave.v2.parser.DocumentParser;
import org.mule.weave.v2.parser.Message;
import org.mule.weave.v2.parser.MessageCollector;
import org.mule.weave.v2.parser.ast.AstNode;
import org.mule.weave.v2.parser.ast.module.ModuleNode;
import org.mule.weave.v2.parser.ast.structure.DocumentNode;
import org.mule.weave.v2.parser.ast.variables.NameIdentifier;
import org.mule.weave.v2.parser.location.WeaveLocation;
import org.mule.weave.v2.parser.phase.*;
import org.mule.weave.v2.runtime.WeaveCompiler;
import org.mule.weave.v2.sdk.*;
import org.mule.weave.v2.utils.AstEmitter;
import org.mule.weave.v2.utils.WeaveFile;
import org.mule.weave.v2.versioncheck.SVersion;
import scala.Option;
import scala.Tuple2;
import scala.collection.Iterator;
import scala.collection.Seq;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static scala.collection.JavaConverters.collectionAsScalaIterable;

@Mojo(name = "compile",
        defaultPhase = LifecyclePhase.COMPILE,
        requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
        requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME,
        executionStrategy = "always")
@Execute(goal = "compile")
public class WeaveCompileMojo extends AbstractWeaveMojo {

    private static final String DW_EXTENSION = "dwl";

    @Parameter(defaultValue = "PARSING", required = false, readonly = false)
    protected ValidationPhase mappingValidation = ValidationPhase.PARSING;

    @Parameter(defaultValue = "FULL", required = false, readonly = false)
    protected ValidationPhase modulesValidation = ValidationPhase.PARSING;

    @Parameter(name = "implicitInputs")
    protected List<String> implicitInputs = new ArrayList<>();

    @Parameter(name = "languageLevel")
    protected String languageLevel = null;

    @Parameter(name = "logDependenciesWarnings", defaultValue = "false")
    protected Boolean logDependenciesWarnings = false;

    @Parameter(name = "binaryCompilation", defaultValue = "false")
    protected Boolean binaryCompilation = false;

    @Parameter(name = "checkDependenciesBinaryCompiled", defaultValue = "false")
    protected Boolean checkDependenciesBinaryCompiled = false;

    @Parameter(name = "disableCommonsSubExpressionElimination", defaultValue = "false")
    protected Boolean disableCommonsSubExpressionElimination = false;

    /**
     * Directory containing the classes and resource files that should be packaged into the JAR.
     */
    @Parameter(defaultValue = "${project.build.outputDirectory}")
    protected File classesDirectory;

    private URL[] classpathUrls;
    private ClassLoaderResourceResolver classLoaderResourceResolver;

    @Override
    public void execute() throws MojoFailureException {
        getLog().info(format("Start Compiling DataWeave: `%s`", project.getName()));
        final List<String> compileClasspathElements = super.compileClasspathElements();
        classpathUrls = compileClasspathElements.stream()
                .map((s) -> {
                    try {
                        return new File(s).toURI().toURL();
                    } catch (MalformedURLException e) {
                        throw new RuntimeException(e);
                    }
                })
                .toArray(URL[]::new);

        try (URLClassLoader resourceClassLoader = new URLClassLoader(classpathUrls, null)) {
            try(URLClassLoader serviceClassLoader = new URLClassLoader(classpathUrls, ModuleLoader.class.getClassLoader())) {
                classLoaderResourceResolver = new ClassLoaderResourceResolver(resourceClassLoader);
                final ChainedWeaveResourceResolver weaveResourceResolver = new ChainedWeaveResourceResolver(
                        collectionAsScalaIterable(
                                Arrays.asList(
                                        new FolderResourceResolver(sourceFolder),
                                        classLoaderResourceResolver)
                        ).toSeq());
                final Seq<ModuleLoader> loaders = collectionAsScalaIterable(singletonList(ModuleLoader.apply(weaveResourceResolver))).toSeq();
                final ModuleParsingPhasesManager moduleParsingPhasesManager = ModuleParsingPhasesManager.apply(ModuleLoaderManager.apply(loaders, () -> {
                    java.util.Iterator<ModuleLoader> iterator = ServiceLoader.load(ModuleLoader.class, serviceClassLoader).iterator();
                    ArrayList<ModuleLoader> modules = new ArrayList<>();
                    while (iterator.hasNext()) {
                        ModuleLoader next = iterator.next();
                        if (next instanceof WeaveResourceResolverAware) {
                            ((WeaveResourceResolverAware) next).resolver(weaveResourceResolver);
                        }
                        if (next instanceof ClassLoaderServiceAware) {
                            ((ClassLoaderServiceAware) next).classLoaderService(className -> {
                                try {
                                    return Option.apply(resourceClassLoader.loadClass(className));
                                } catch (ClassNotFoundException e) {
                                    return Option.empty();
                                }
                            });
                        }
                        modules.add(next);
                    }
                    return collectionAsScalaIterable(modules).toSeq();
                }));
                final int numberOfError = validate(sourceFolder, "", moduleParsingPhasesManager);
                moduleParsingPhasesManager.invalidateAll();
                if (numberOfError > 0) {
                    throw new MojoFailureException(format("Compilation failure %s errors found.", numberOfError));
                }
            }
        } catch (IOException e) {
            //
        } finally {
            classLoaderResourceResolver = null;
        }
        getLog().info(format("DataWeave `%s` successfully compiled", project.getName()));
    }

    private int validate(File sourceDir, String name, ModuleParsingPhasesManager moduleParsingPhasesManager) {
        int errors = 0;
        final File[] child = Optional.ofNullable(sourceDir.listFiles()).orElse(new File[0]);
        for (File file : child) {
            if (isDWFile(file)) {
                getLog().debug(format("Compiling %s", file.getAbsolutePath()));
                final String basename = FilenameUtils.getBaseName(file.getName());
                errors += parse(NameIdentifier.apply(nameIdentifier(name, basename), Option.empty()), file, moduleParsingPhasesManager);
            } else if (file.isDirectory()) {
                errors += validate(file, nameIdentifier(name, file.getName()), moduleParsingPhasesManager);
            }
        }
        return errors;
    }

    private boolean isDWFile(File file) {
        return Optional.ofNullable(file).map(f -> DW_EXTENSION.equals(FilenameUtils.getExtension(f.getName()))).orElse(false);
    }

    private String nameIdentifier(String name, String moduleName) {
        if (name.isEmpty()) {
            return moduleName;
        }
        return name + NameIdentifier.SEPARATOR() + moduleName;
    }

    private int parse(NameIdentifier identifier, File file, ModuleParsingPhasesManager moduleParsingPhasesManager) {
        final DocumentParser documentParser = new DocumentParser(15, NoInspector$.MODULE$);
        final ParsingContext parsingContext = ParsingContextFactory.createParsingContext(identifier, moduleParsingPhasesManager);
        for (String implicitInput : implicitInputs) {
            parsingContext.addImplicitInput(implicitInput, Option.empty());
        }
        if (languageLevel != null) {
            Option<SVersion> sVersionOption = SVersion.fromString(languageLevel);
            parsingContext.languageLevel_$eq(sVersionOption);
        }
        final WeaveResource input = WeaveResourceFactory.fromFile(file);
        final MessageCollector messages = compile(identifier, documentParser, input, file, parsingContext);
        final Iterator<Tuple2<WeaveLocation, Message>> errors = messages.errorMessages().toIterator();
        while (errors.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = errors.next();
            final WeaveLocation weaveLocation = next._1;
            try {
                getLog().error(format("%s: [%s, %s]\n%s at %s",
                        file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column(),
                        next._2.message(), weaveLocation.locationString()));
            } catch (IOException ignored) {
                // Nothing to do
            }
        }

        final Iterator<Tuple2<WeaveLocation, Message>> warnings = messages.warningMessages().toIterator();
        while (warnings.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = warnings.next();
            final WeaveLocation weaveLocation = next._1;
            final File source = new File(sourceFolder, NameIdentifierHelper.toWeaveFilePath(weaveLocation.resourceName()));
            final boolean messageFromDependency = !source.exists();
            if (!messageFromDependency || Boolean.TRUE.equals(logDependenciesWarnings)) {
                try {
                    getLog().warn(format("%s: [%s, %s]\n%s at %s",
                            file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column(),
                            next._2.message(), weaveLocation.locationString()));
                } catch (IOException ignored) {
                    // Nothing to do
                }
            }
        }

        return messages.errorMessages().length();
    }

    private MessageCollector compile(NameIdentifier identifier, DocumentParser documentParser, WeaveResource input, File inputFile, ParsingContext parsingContext) {
        final PhaseResult<ParsingResult<AstNode>> parseResult = documentParser.parse(input, parsingContext);
        MessageCollector messages;
        if (parseResult.hasResult()) {
            ParsingResult<AstNode> result = parseResult.getResult();
            if (result.astNode() instanceof ModuleNode) {
                if (modulesValidation == ValidationPhase.FULL) {
                    PhaseResult<TypeCheckingResult<? extends AstNode>> typeCheckingResultPhaseResult = documentParser.runAllPhases(input, parsingContext);
                    messages = typeCheckingResultPhaseResult.messages();
                } else {
                    messages = parseResult.messages();
                }
            } else {
                if (mappingValidation == ValidationPhase.FULL) {
                    PhaseResult<TypeCheckingResult<? extends AstNode>> typeCheckingResultPhaseResult = documentParser.runAllPhases(input, parsingContext);
                    messages = typeCheckingResultPhaseResult.messages();
                } else {
                    messages = parseResult.messages();
                }
            }

            if (Boolean.TRUE.equals(binaryCompilation) && !messages.hasErrors()) {
                messages = binaryCompile(identifier, inputFile, input, result.astNode(), parsingContext);
            }
        } else {
            messages = parseResult.messages();
        }
        return messages;
    }

    private MessageCollector binaryCompile(NameIdentifier identifier, File inputFile, WeaveResource input, AstNode parserAstNode, ParsingContext parsingContext) {
        getLog().info("Generating binary compilation file for resource: " + inputFile.getAbsolutePath());
        MessageCollector messages = MessageCollector.apply();
        if (!classesDirectory.exists()) {
            messages.addErrorValidationMessage(new ValidationMessage(Message.apply("BinaryCompilationMissingTargetFolder", format("Could not generate binary compilation file for resource: %s due to classesDirectory doesn't exist: '%s'", inputFile.getAbsolutePath(), classesDirectory), () -> "Compilation"),
                    parserAstNode.location()));
        } else {
            if (Boolean.TRUE.equals(disableCommonsSubExpressionElimination)) {
                parsingContext.disableCommonSubExpressionElimination();
            }
            if (parserAstNode instanceof ModuleNode) {
                PhaseResult<ScopeGraphResult<ModuleNode>> preCompilationResultPhaseResult;
                preCompilationResultPhaseResult = WeaveCompiler.preCompileModule(input, parsingContext);
                messages = preCompilationResultPhaseResult.messages();
                List<NameIdentifier> modulesWithoutBinaryCompilation = classLoaderResourceResolver.getModulesWithoutBinaryCompilation();
                if (Boolean.TRUE.equals(checkDependenciesBinaryCompiled) && !modulesWithoutBinaryCompilation.isEmpty()) {
                    messages.addErrorValidationMessage(new ValidationMessage(Message.apply("BinaryCompilationMissingBinaryModuleDependency", format("There was an error compiling to binary resource: '%s', it depends on modules: %s which are not binary compiled.", identifier, modulesWithoutBinaryCompilation), () -> "Compilation"),
                            parserAstNode.location()));
                }
                if (!preCompilationResultPhaseResult.hasErrors() && preCompilationResultPhaseResult.hasResult()) {
                    AstNode compileNode = preCompilationResultPhaseResult.getResult().astNode();
                    generateValidateBinaryFile(identifier, inputFile, input, compileNode, messages);
                }
            } else {
                PhaseResult<ScopeGraphResult<DocumentNode>> preCompilationResultPhaseResult;
                preCompilationResultPhaseResult = WeaveCompiler.preCompile(input, parsingContext);
                messages = preCompilationResultPhaseResult.messages();
                List<NameIdentifier> modulesWithoutBinaryCompilation = classLoaderResourceResolver.getModulesWithoutBinaryCompilation();
                if (Boolean.TRUE.equals(checkDependenciesBinaryCompiled) && !modulesWithoutBinaryCompilation.isEmpty()) {
                    messages.addErrorValidationMessage(new ValidationMessage(Message.apply("BinaryCompilationMissingBinaryModuleDependency", format("There was an error compiling to binary resource: '%s', it depends on modules: %s which are not binary compiled.", identifier,
                            modulesWithoutBinaryCompilation.stream().map(NameIdentifier::name).collect(Collectors.joining(", ", "[", "]"))), () -> "Compilation"),
                            parserAstNode.location()));
                }
                if (!preCompilationResultPhaseResult.hasErrors() && preCompilationResultPhaseResult.hasResult()) {
                    AstNode compileNode = preCompilationResultPhaseResult.getResult().astNode();
                    generateValidateBinaryFile(identifier, inputFile, input, compileNode, messages);
                }
            }
        }
        return messages;
    }

    private void generateValidateBinaryFile(NameIdentifier identifier, File inputFile, WeaveResource input, AstNode compileNode, MessageCollector messages) {
        File binaryFile = WeaveBinaryCompiler.toBinary(compileNode, inputFile, sourceFolder, classesDirectory);
        AstNode deserializedNode = WeaveBinaryCompiler.toAstNode(binaryFile, input, identifier);

        getLog().debug("Validating binary compilation AST for input: " + inputFile.getAbsolutePath());
        AstEmitter astEmitter = new AstEmitter(true, false, true, true);
        String deserializedAstNodeString = astEmitter.print(deserializedNode);
        String compiledNodeString = astEmitter.print(compileNode);
        if (!deserializedAstNodeString.equals(compiledNodeString)) {
            getLog().debug("Compiled AST node: \n" + compiledNodeString);
            getLog().debug("Binary deserialized compiled AST node: \n" + deserializedAstNodeString);
            messages.addErrorValidationMessage(new ValidationMessage(Message.apply("BinaryCompilationDoNotMatchASTFromSource", "The AST after being compiled to binary is not matching the AST from parsing phase. This is a bug.", () -> "Compilation"),
                    compileNode.location()));
        }
    }

    private static class ClassLoaderResourceResolver implements WeaveResourceResolver {

        private final URLClassLoader urlClassLoader;
        private List<NameIdentifier> missingBinaryCompiledModules = new ArrayList<>();

        public ClassLoaderResourceResolver(URLClassLoader urlClassLoader) {
            this.urlClassLoader = urlClassLoader;
        }

        @Override
        public Option<WeaveResource> resolve(NameIdentifier name) {
            String weaveFilePath = NameIdentifierHelper.toWeaveFilePath(name, "/");
            checkBinaryCompiledFile(name, weaveFilePath);
            return resolvePath(weaveFilePath);
        }

        @Override
        public Option<WeaveResource> resolvePath(String path) {
            String weaveFilePath = path.startsWith("/") ? path.substring(1) : path;
            Option<URL> maybeUrl = resolveUrl(weaveFilePath);
            return maybeUrl.map(WeaveResourceFactory::fromUrl);
        }

        private void checkBinaryCompiledFile(NameIdentifier name, String path) {
            String weaveFilePath = path.startsWith("/") ? path.substring(1) : path;
            if (resolveUrl(weaveFilePath.replace(WeaveFile.fileExtension(), WeaveFile.binaryFileExtension())).isEmpty()) {
                missingBinaryCompiledModules.add(name);
            }
        }

        private Option<URL> resolveUrl(String weaveFilePath) {
            return Option.apply(urlClassLoader.getResource(weaveFilePath));
        }

        public List<NameIdentifier> getModulesWithoutBinaryCompilation() {
            return missingBinaryCompiledModules.stream().distinct().collect(Collectors.toList());
        }

    }

    private static class FolderResourceResolver implements WeaveResourceResolver {

        private final File sourceFolder;

        public FolderResourceResolver(File sourceFolder) {
            this.sourceFolder = sourceFolder;
        }

        @Override
        public Option<WeaveResource> resolve(NameIdentifier name) {
            String weaveFilePath = NameIdentifierHelper.toWeaveFilePath(name, "/");
            return resolvePath(weaveFilePath);
        }

        @Override
        public Option<WeaveResource> resolvePath(String path) {
            String weaveFilePath = path.startsWith("/") ? path.substring(1) : path;
            File resource = new File(sourceFolder, weaveFilePath);
            if (resource.exists()) {
                FileInputStream input = null;
                try {
                    input = new FileInputStream(resource);
                    String content = IOUtils.toString(input, StandardCharsets.UTF_8);
                    return Option.apply(WeaveResourceFactory.fromContent(resource.toURI().toURL().toExternalForm(), content));
                } catch (IOException e) {
                    return Option.empty();
                } finally {
                    try {
                        if (input != null) {
                            input.close();
                        }
                    } catch (IOException e) {
                        //
                    }
                }
            } else {
                return Option.empty();
            }
        }

        @Override
        public Seq<WeaveResource> resolveAll(NameIdentifier name) {
            return resolve(name).toList();
        }
    }
}
