package org.mule.weave.maven.plugin;


import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.cli.Commandline;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.Map;
import java.util.Properties;

import static java.lang.String.format;
import static java.lang.System.getProperty;

public class WeaveTestCommandlineRequestBuilder {

    private final MavenProject project;
    private final String classpath;
    private final String[] args;
    private final WeaveTestMojoConfig mojoConfig;

    public WeaveTestCommandlineRequestBuilder(MavenProject project, String classpath, String[] args, WeaveTestMojoConfig mojoConfig) {
        this.project = project;
        this.classpath = classpath;
        this.args = args;
        this.mojoConfig = mojoConfig;
    }

    public Commandline build() {
        final Commandline cli = new Commandline();
        cli.setWorkingDirectory(project.getBasedir());
        cli.setExecutable("java");

        setupEnvironmentVariables(cli);

        // Set up system properties
        if (mojoConfig.getRunnerSystemProperties() != null) {
            for (final Map.Entry<String, String> entry : mojoConfig.getRunnerSystemProperties().entrySet()) {
                cli.createArg().setValue(format("-D%s=%s", entry.getKey(), entry.getValue()));
            }
        }
        //Test system property send it as testToRun to the testing engine
        String test = getProperty("test");
        if (test != null) {
            cli.createArg().setValue(format("-D%s=%s", "testToRun", test));
        }

        if (mojoConfig.isRunnerJvmDebug()) {
            cli.createArg().setValue("-agentlib:jdwp=transport=dt_socket,server=y,suspend=y,address=5005");
        }

        cli.createArg().setValue(format("-Dbasedir=%s", project.getBasedir().getAbsolutePath()));

        // Set user specified JVM arguments
        if (mojoConfig.getRunnerArgLine() != null) {
            cli.createArg().setLine(mojoConfig.getRunnerArgLine());
        }

        // Set ScalaTest arguments
        cli.createArg().setValue("org.mule.weave.v2.module.test.runner.TestRunner");

        if (args != null) {
            for (final String arg : args) {
                cli.createArg().setValue(arg);
            }
        }
        return cli;
    }

    /**
     * Configures the EnvironmentVariables for the CLI
     * Explicit envVars overrides the File Based ones.
     * @param cli Commandline to parameterize.
     */
    private void setupEnvironmentVariables(Commandline cli){
        // Set up environment variables
        //file sourced environment variables.
        if(mojoConfig.getRunnerEnvVarFile()!=null && mojoConfig.getRunnerEnvVarFile().exists()){
            try(InputStream is = Files.newInputStream(mojoConfig.getRunnerEnvVarFile().toPath())) {
                Properties properties = new Properties();
                properties.load(is);
                for (final Map.Entry<Object, Object> entry : properties.entrySet()) {
                    cli.addEnvironment(entry.getKey().toString(), entry.getValue().toString());
                }
            }catch(IOException ex){
                throw new IllegalArgumentException("Error reading envVarFile: " + mojoConfig.getRunnerEnvVarFile(), ex);
            }
        }

        //standalone env vars - overwrites the file based ones.
        if (mojoConfig.getRunnerEnvironmentVariables() != null) {
            for (final Map.Entry<String, String> entry : mojoConfig.getRunnerEnvironmentVariables().entrySet()) {
                cli.addEnvironment(entry.getKey(), entry.getValue());
            }
        }

        cli.addEnvironment("CLASSPATH", classpath);
    }

}
