package org.mule.weave.maven.plugin;

import static java.lang.String.format;
import static com.mulesoft.weave.docs.DocTemplateBundles.exchangeMarkdown;
import static com.mulesoft.weave.docs.DocTemplateBundles.templateByName;
import static com.mulesoft.weave.docs.WeaveDocsGenerator.generate;

import com.mulesoft.weave.docs.DocTemplateBundle;
import com.mulesoft.weave.docs.WeaveDocsGenerator;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.ResolutionScope;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Mojo(name = "generate-docs",
        defaultPhase = LifecyclePhase.PREPARE_PACKAGE,
        requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
        requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME,
        threadSafe = true)
@Execute(goal = "generate-docs")
public class WeaveGenerateDocsMojo extends WeaveDocsMojo {
    private static final Pattern VERSION_PATTER = Pattern.compile("([1-9]\\d*)\\.(\\d+)\\.(\\d+)(?:-([a-zA-Z0-9]+))?");
    private static final String WLANG_GROUP_ID = "org.mule.weave";
    private static final String WLANG_ARTIFACT_ID = "wlang";
    
    @Override
    protected void doExecute() throws MojoExecutionException {
        try {
            final File canonicalFile = config.getOutput().getCanonicalFile();
            getLog().info(format("Generating docs at: %s", canonicalFile));
            if (!config.getOutput().exists()) {
                // Make sure it exists
                config.getOutput().mkdirs();
            }

            final Map<String, String> docsConfiguration = new HashMap<>();
            final Optional<Artifact> maybeWlangArtifact = project.getArtifacts()
                    .stream()
                    .filter(artifact -> WLANG_GROUP_ID.equals(artifact.getGroupId()) && WLANG_ARTIFACT_ID.equals(artifact.getArtifactId()))
                    .findFirst();
            if (maybeWlangArtifact.isPresent()) {
                Artifact wlangArtifact = maybeWlangArtifact.get();
                String wlangVersion = wlangArtifact.getBaseVersion();
                final Matcher matcher = VERSION_PATTER.matcher(wlangVersion);
                if (matcher.matches()) {
                    String major = matcher.group(1);
                    String minor = matcher.group(2);
                    docsConfiguration.put(WeaveDocsGenerator.MIN_DW_VERSION(), format("%s.%s", major, minor));
                } else {
                    getLog().warn(format("Invalid wlang version: %s", wlangVersion));
                }
            } else {
                getLog().warn("Enable to find wlang artifact.");
            }
            final DocTemplateBundle docTemplateBundleOption = templateByName(config.getTemplate(), exchangeMarkdown());
            if (config.getHomePage() != null) {
                docsConfiguration.put(WeaveDocsGenerator.MODULE_DOC_PATH(), config.getHomePage().getPath());
            }
            generate(docTemplateBundleOption, sourceFolder, canonicalFile, docsConfiguration);
            getLog().info(format("Generated docs at: %s", canonicalFile));
        } catch (IOException e) {
            throw new MojoExecutionException("Exception while resolving.", e);
        }
    }
}
