package org.mule.weave.maven.plugin;

import org.apache.commons.io.IOUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.*;
import org.codehaus.plexus.util.FileUtils;
import org.mule.weave.v2.inspector.NoInspector$;
import org.mule.weave.v2.parser.DocumentParser;
import org.mule.weave.v2.parser.Message;
import org.mule.weave.v2.parser.MessageCollector;
import org.mule.weave.v2.parser.ast.AstNode;
import org.mule.weave.v2.parser.ast.module.ModuleNode;
import org.mule.weave.v2.parser.ast.variables.NameIdentifier;
import org.mule.weave.v2.parser.location.WeaveLocation;
import org.mule.weave.v2.parser.phase.*;
import org.mule.weave.v2.sdk.*;
import org.mule.weave.v2.versioncheck.SVersion;
import scala.Option;
import scala.Tuple2;
import scala.collection.Iterator;
import scala.collection.Seq;
import scala.collection.Seq$;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static java.lang.String.format;

@Mojo(name = "compile",
        defaultPhase = LifecyclePhase.COMPILE,
        requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
        requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME,
        executionStrategy = "always")
@Execute(goal = "compile")
public class WeaveCompileMojo extends AbstractWeaveMojo {

    private static final String DW_EXTENSION = "dwl";

    @Parameter(defaultValue = "PARSING", required = false, readonly = false)
    protected ValidationPhase mappingValidation = ValidationPhase.PARSING;

    @Parameter(defaultValue = "FULL", required = false, readonly = false)
    protected ValidationPhase modulesValidation = ValidationPhase.PARSING;

    @Parameter(name = "implicitInputs")
    protected List<String> implicitInputs = new ArrayList<>();

    @Parameter(name = "languageLevel")
    protected String languageLevel = null;

    @Override
    public void execute() throws MojoFailureException {
        getLog().info(format("Start Compiling DataWeave: `%s`", project.getName()));
        final List<String> strings = super.compileClasspathElements();
        URL[] urls = strings.stream().map((s) -> {
            try {
                return new File(s).toURI().toURL();
            } catch (MalformedURLException e) {
                throw new RuntimeException(e);
            }
        }).toArray(URL[]::new);
        URLClassLoader urlClassLoader = null;
        try {
            urlClassLoader = new URLClassLoader(urls, null);
            final ChainedWeaveResourceResolver weaveResourceResolver = new ChainedWeaveResourceResolver(
                    Seq$.MODULE$.<WeaveResourceResolver>newBuilder()
                            .$plus$eq(new ClassLoaderResourceResolver(urlClassLoader))
                            .$plus$eq(new FolderResourceResolver(sourceFolder))
                            .result());
            final Seq<ModuleLoader> loaders = Seq$.MODULE$.<ModuleLoader>newBuilder().$plus$eq(ModuleLoader$.MODULE$.apply(weaveResourceResolver)).result();
            final ModuleParsingPhasesManager moduleParsingPhasesManager = ModuleParsingPhasesManager$.MODULE$.apply(ModuleLoaderManager.apply(loaders));
            final int numberOfError = validate(sourceFolder, "", moduleParsingPhasesManager);
            if (numberOfError > 0) {
                throw new MojoFailureException(format("Compilation failure %s errors found.", numberOfError));
            }
        } finally {
            if (urlClassLoader != null) {
                try {
                    urlClassLoader.close();
                } catch (IOException e) {
                    //
                }
            }
        }
        getLog().info(format("DataWeave `%s` successfully compiled", project.getName()));
    }

    private int validate(File sourceDir, String name, ModuleParsingPhasesManager moduleParsingPhasesManager) {
        int errors = 0;
        final File[] child = Optional.ofNullable(sourceDir.listFiles()).orElse(new File[0]);
        for (File file : child) {
            if (isDWFile(file)) {
                getLog().debug(format("Compiling %s", file.getAbsolutePath()));
                final String basename = FileUtils.basename(file.getName());
                errors += parse(NameIdentifier.apply(nameIdentifier(name, basename), Option.empty()), file, moduleParsingPhasesManager);
            } else if (file.isDirectory()) {
                validate(file, nameIdentifier(name, file.getName()), moduleParsingPhasesManager);
            }
        }
        return errors;
    }

    private boolean isDWFile(File file) {
        return Optional.ofNullable(file).map(f -> DW_EXTENSION.equals(FileUtils.extension(f.getName()))).orElse(false);
    }

    private String nameIdentifier(String name, String moduleName) {
        if (name.isEmpty()) {
            return moduleName;
        }
        return name + NameIdentifier.SEPARATOR() + moduleName;
    }

    private int parse(NameIdentifier identifier, File file, ModuleParsingPhasesManager moduleParsingPhasesManager) {
        final DocumentParser documentParser = new DocumentParser(15, NoInspector$.MODULE$);
        final ParsingContext parsingContext = ParsingContextFactory.createParsingContext(identifier, moduleParsingPhasesManager);
        for (String implicitInput : implicitInputs) {
            parsingContext.addImplicitInput(implicitInput, Option.empty());
        }
        if (languageLevel != null) {
            Option<SVersion> sVersionOption = SVersion.fromString(languageLevel);
            parsingContext.languageLevel_$eq(sVersionOption);
        }
        final WeaveResource input = WeaveResourceFactory.fromFile(file);
        final MessageCollector messages = compile(documentParser, input, parsingContext);
        final Iterator<Tuple2<WeaveLocation, Message>> errors = messages.errorMessages().toIterator();
        while (errors.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = errors.next();
            final WeaveLocation weaveLocation = next._1;
            try {
                getLog().error(format("%s: [%s, %s]\n%s at %s",
                        file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column(),
                        next._2.message(), weaveLocation.locationString()));
            } catch (IOException ignored) {
                // Nothing to do
            }
        }

        final Iterator<Tuple2<WeaveLocation, Message>> warnings = messages.warningMessages().toIterator();
        while (warnings.hasNext()) {
            final Tuple2<WeaveLocation, Message> next = warnings.next();
            final WeaveLocation weaveLocation = next._1;
            try {
                getLog().warn(format("%s: [%s, %s]\n%s at %s",
                        file.getCanonicalPath(), weaveLocation.startPosition().line(), weaveLocation.startPosition().column(),
                        next._2.message(), weaveLocation.locationString()));
            } catch (IOException ignored) {
                // Nothing to do
            }
        }

        return messages.errorMessages().length();
    }

    private MessageCollector compile(DocumentParser documentParser, WeaveResource input, ParsingContext parsingContext) {
        final PhaseResult<ParsingResult<AstNode>> parseResult = documentParser.parse(input, parsingContext);
        MessageCollector messages;
        if (parseResult.hasResult()) {
            ParsingResult<AstNode> result = parseResult.getResult();
            if (result.astNode() instanceof ModuleNode) {
                if (modulesValidation == ValidationPhase.FULL) {
                    PhaseResult<TypeCheckingResult<? extends AstNode>> typeCheckingResultPhaseResult = documentParser.runAllPhases(input, parsingContext);
                    messages = typeCheckingResultPhaseResult.messages();
                } else {
                    messages = parseResult.messages();
                }
            } else {
                if (mappingValidation == ValidationPhase.FULL) {
                    PhaseResult<TypeCheckingResult<? extends AstNode>> typeCheckingResultPhaseResult = documentParser.runAllPhases(input, parsingContext);
                    messages = typeCheckingResultPhaseResult.messages();
                } else {
                    messages = parseResult.messages();
                }
            }
        } else {
            messages = parseResult.messages();
        }
        return messages;
    }

    private static class ClassLoaderResourceResolver implements WeaveResourceResolver {

        private final URLClassLoader urlClassLoader;

        public ClassLoaderResourceResolver(URLClassLoader urlClassLoader) {
            this.urlClassLoader = urlClassLoader;
        }

        @Override
        public Option<WeaveResource> resolve(NameIdentifier name) {
            String weaveFilePath = NameIdentifierHelper.toWeaveFilePath(name, "/");
            return resolvePath(weaveFilePath);
        }

        @Override
        public Option<WeaveResource> resolvePath(String path) {
            String weaveFilePath = path.startsWith("/") ? path.substring(1) : path;
            URL resource = urlClassLoader.getResource(weaveFilePath);
            if (resource != null) {
                try {
                    String content = IOUtils.toString(resource.openStream(), StandardCharsets.UTF_8);
                    return Option.apply(WeaveResource.apply(resource.toExternalForm(), content));
                } catch (IOException e) {
                    return Option.empty();
                }
            } else {
                return Option.empty();
            }
        }

        @Override
        public Seq<WeaveResource> resolveAll(NameIdentifier name) {
            return resolve(name).toList();
        }
    }

    private static class FolderResourceResolver implements WeaveResourceResolver {

        private final File sourceFolder;

        public FolderResourceResolver(File sourceFolder) {
            this.sourceFolder = sourceFolder;
        }

        @Override
        public Option<WeaveResource> resolve(NameIdentifier name) {
            String weaveFilePath = NameIdentifierHelper.toWeaveFilePath(name, "/");
            return resolvePath(weaveFilePath);
        }

        @Override
        public Option<WeaveResource> resolvePath(String path) {
            String weaveFilePath = path.startsWith("/") ? path.substring(1) : path;
            File resource = new File(sourceFolder, weaveFilePath);
            if (resource.exists()) {
                FileInputStream input = null;
                try {
                    input = new FileInputStream(resource);
                    String content = IOUtils.toString(input, StandardCharsets.UTF_8);
                    return Option.apply(WeaveResource.apply(resource.toURI().toURL().toExternalForm(), content));
                } catch (IOException e) {
                    return Option.empty();
                } finally {
                    try {
                        input.close();
                    } catch (IOException e) {
                        //
                    }
                }
            } else {
                return Option.empty();
            }
        }

        @Override
        public Seq<WeaveResource> resolveAll(NameIdentifier name) {
            return resolve(name).toList();
        }
    }
}
