package org.mule.weave.maven.plugin.client;

import static java.net.URLEncoder.encode;
import static java.nio.charset.StandardCharsets.UTF_8;
import static javax.ws.rs.client.ClientBuilder.newBuilder;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.Response.Status.Family.SUCCESSFUL;
import static javax.ws.rs.core.Response.Status.Family.familyOf;

import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import org.mule.weave.maven.plugin.exchange.exception.ClientException;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.Response;
import java.io.UnsupportedEncodingException;

import java.util.function.Consumer;

public abstract class RestClient {
    
    private static final String PATCH = "PATCH";
    
    private final boolean withMultiPartFeature;
    private final boolean suppressHttpComplianceValidation;

    public RestClient(boolean withMultiPartFeature, boolean suppressHttpComplianceValidation) {
        this.withMultiPartFeature = withMultiPartFeature;
        this.suppressHttpComplianceValidation = suppressHttpComplianceValidation;
    }

    public Response postFormUrlEncoded(String uri, String path, Entity<Form> form) {
        Response response = builder(uri, path).post(form);
        checkResponseSuccessfulStatus(response);
        return response;
    }

    public <U> U postFormUrlEncoded(String uri, String path, Entity<Form> form, Class<U> clazz) {
        return postFormUrlEncoded(uri, path, form).readEntity(clazz);
    }

    public <T> Response post(String uri, String path, Entity<T> entity) {
        Response response = builder(uri, path).post(entity);
        checkResponseSuccessfulStatus(response);
        return response;
    }
    
    public <T> Response post(String uri, String path, T entity) {
        return post(uri, path, Entity.json(entity));
    }
    
    public <T, U> U post(String uri, String path, T entity, Class<U> clazz) {
        return post(uri, path, entity).readEntity(clazz);
    }

    public <T> Response put(String uri, String path, Entity<T> entity, Consumer<Invocation.Builder> requestCustomizer) {
        Invocation.Builder builder = builder(uri, path);
        requestCustomizer.accept(builder);
        Response response = builder.put(entity);
        checkResponseSuccessfulStatus(response);
        return response;
    }

    public <T> Response patch(String uri, String path, Entity<T> entity, Consumer<Invocation.Builder> requestCustomizer) {
        Invocation.Builder builder = builder(uri, path);
        requestCustomizer.accept(builder);
        Response response = builder.method(PATCH, entity);
        checkResponseSuccessfulStatus(response);
        return response;
    }

    public <T> Response patch(String uri, String path, T entity, Consumer<Invocation.Builder> requestCustomizer) {
        return patch(uri, path, Entity.json(entity), requestCustomizer);
    }
    
    public String encodeUtf8(String value) {
        try {
            return encode(value, UTF_8.toString());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public void checkResponseSuccessfulStatus(Response response) throws ClientException {
        if (familyOf(response.getStatus()) != SUCCESSFUL) {
            throw new ClientException(response);
        }
    }

    private Invocation.Builder builder(String uri, String path) {
        Client client = newBuilder()
                .property(ClientProperties.CONNECTOR_PROVIDER, "org.glassfish.jersey.netty.connector.NettyConnectorProvider")
                .build();
        if (withMultiPartFeature) {
            client.register(MultiPartFeature.class);
        }
        if (suppressHttpComplianceValidation) {
            client.property(ClientProperties.SUPPRESS_HTTP_COMPLIANCE_VALIDATION, true);
        }

        WebTarget target = client.target(uri).path(path);
        return target.request(APPLICATION_JSON_TYPE);
    }
}
