%dw 2.0

import * from dw::io::file::FileSystem
import run, EvalSuccess, ExecutionFailure, fail from dw::Runtime
import * from dw::test::Tests
import * from dw::test::Asserts
import buildContext from dw::test::internal::Functions

type TestResourceProvider = (mappingDir: String, testCase: String) -> {content: String, url: String}

/**
* Creates all the tests suites from a given directory.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | directory | The directory where to search all the test cases
* |===
*
**/
fun createTestSuite(directory: String, resourceProvider: TestResourceProvider): Array<() -> TestResult> = do {
    var mappings = ls(directory)
    ---
    if(isEmpty(mappings))
        []
    else
        flatten(mappings map ((mappingDir) ->
            createTestCases(mappingDir, resourceProvider)
        ))
}

/**
* Creates a Single Test Case from the
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | mappingDir | The mapping directory path
* | testUrlProvider | Factory function that returns the url of the mapping file to be executed
* |===
*
**/
fun createTestCases(mappingDir: String, resourceProvider: TestResourceProvider): Array<() -> TestResult> = do {
    ls(mappingDir) //Filter to check there is an out if not ignore the folder
        orderBy ((testCase, index) -> nameOf(testCase))
        filter ((testCase) -> !isEmpty(ls(testCase, "out.*")))
        flatMap ((testCase) -> do {
            var testCaseName: String = nameOf(testCase)
            var mappingUrl =  resourceProvider(mappingDir, testCaseName)
            var context: { _*: Any } = buildContext(path(testCase, "inputs"))
            var outs = ls(testCase, "out.*")
                           map ((expectedPath) -> do {
                                  var mimeType = mimeTypeOf(expectedPath) default "text/plain"
                                  ---
                                  {
                                    path: expectedPath,
                                    value: read(contentOf(expectedPath), mimeType),
                                    mimeType: mimeType
                                  }
                                }
                              )
            ---
            outs map ((expected) -> do {
              createTestCase(nameOf(mappingDir) ++ " - " ++ testCaseName, mappingUrl, context, expected.value, expected.path, expected.mimeType)
            })
          })
}

fun evalPathTest(testUrl: {content: String, url: String}, context: {_?: Any}, expected: Any, expectedPath: String, mimeType: String) : TestResult = do {
              var resultValue = run(testUrl.url,
                                              {(testUrl.url): testUrl.content},
                                              {},
                                              context,
                                      {
                                                    outputMimeType: mimeType,
                                                      onException: "FAIL",
                                                      loggerService: {
                                                          log: (level, msg, ctx)-> log(msg)
                                                      }
                                                    }
                                                         ).value

              var testResult = read(resultValue,mimeType) must equalTo(expected)
              ---
              if(dw::Runtime::prop("updateResult") == "true")
                  testResult match {
                      case m is {"matches": true} -> m
                      else -> do {
                          var logResult = log("Updating Test Result '$(expectedPath)'")
                          var amount = copyTo(resultValue, expectedPath)
                          ---
                          $
                      }
                  }
              else
                  testResult
}

fun createTestCase(scenario: String, testUrl: {content: String, url: String}, context: {_?: Any}, expected: Any, expectedPath: String, mimeType: String): () -> TestResult = do {
    () -> scenario in do {
            evalPathTest(testUrl,context,expected,expectedPath,mimeType)
    }
}
