/**
*
* This module contains all functions required to create a DataWeave test
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
* ---
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
*      "It should support simple matching" in do {
*          var payload = {}
*          ---
*          payload must beObject()
*      },
*
*      "It should support multiple root cases" in do {
*          var payload = {}
*          var flowVar = {a: 123}
*          ---
*          [
*              payload must beObject(),
*              flowVar must [
*                  beObject(),
*                  $.a must equalTo(123)
*              ]
*          ]
*      },
*      "It should support using custom assertions" in  do {
*          var payload = []
*          ---
*          payload must sizeOf($) > 2
*      }
*  ]
* ----
*/
%dw 2.5

import mergeWith from dw::core::Objects
import * from dw::io::file::FileSystem
import * from dw::util::Timer
import some from dw::core::Arrays
import * from dw::test::Asserts
import interceptor, buildContext from dw::test::internal::Functions

import location, Location, run, TryResult, eval, TryResultFailure, try from dw::Runtime


@AnnotationTarget(targets = ["Expression"])
annotation Skip(reason: String = "skipped")

var ERROR_STATUS = "ERROR"
var OK_STATUS    = "OK"
var FAIL_STATUS  = "FAIL"
var SKIP_STATUS  = "SKIP"

/**
* Data Type that describes the result of a Test Execution
*/
type TEST_STATUS = "ERROR" | "OK" | "FAIL" | "SKIP"

type TestResult = {
    name: String,
    time: Number,
    status: TEST_STATUS,
    tests?: Array<TestResult>,
    errorMessage?: String,
    skipReason?: String,
    location?: Location,
}

type TestConfig<Ctx <: Object> = {|
  setup: () -> Ctx,
  teardown: (c: Ctx) -> Any
|}

/**
* Generates configuration for a test(s) that needs setup/teardown stages. Intended for it to used in combination with
* the `in` function.
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*
* var config = {
*   setup: () -> { contextString: "context", otherContext: 3 },
*   teardown: () -> {}
* }
* ---
* "It should generate context for following tests" withConfig config in  [
*   do { $.contextString must beString() },
*   do { $.otherContext must equalTo(3) }
* ]
* ----
*/
fun withConfig<Ctx <: Object>(testName: String, config: TestConfig<Ctx>) = {
  config: config,
  testName: testName
}

/**
* Defines a new test case inside a test suite with its relevant context.
* Intended to be used in combination with withConfig
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*
* var config = {
*   setup: () -> { contextString: "context" },
*   teardown: () -> {}
* }
* ---
* "It should generate context for following tests" withConfig config in do {
*   $.contextString must beString()
*  }
*
* ----
*/
fun in<Ctx <: Object>(testSetup: { config: TestConfig<Ctx>, testName: String }, test: (c: Ctx) -> MatcherResult): TestResult = in(testSetup, [test])


/**
* Defines multiple new test cases inside a test suite that share the same context.
* Intended to be used in combination with withConfig
*
* === Example
*
* ==== Source
* [source,DataWeave,linenums]
* ----
*
* var config = {
*   setup: () -> { contextString: "context" },
*   teardown: () -> {}
* }
* ---
* "It should generate context for following tests" withConfig config in  [
*   do { $.contextString must beString() },
*   do { $.otherContext must equalTo(3) }
* ]
* ----
*/
fun in<Ctx <: Object>(testSetup: { config: TestConfig<Ctx>, testName: String }, test: Array<(c: Ctx) -> MatcherResult>): TestResult = do {
  var skipAll: Boolean = (dw::Runtime::prop("skipAll") default false) as Boolean
  var testToRun: String | Null = dw::Runtime::prop("testToRun")
  var patternMatches = testMatches(testToRun, testSetup.testName)
  ---
  if (testSetup.testName.^skip is Null) do {
    var maybeCtx = try(() -> testSetup.config.setup())
    ---
    maybeCtx match {
    	case is { success: true, result: Ctx } -> do {
    	    var result = in(testSetup.testName, test, $.result, skipAll)
          var ignore = try(() -> testSetup.config.teardown($.result))
          ---
          result
    	}
    	case is { success: false, error: TryResultFailure } -> do {
    	  var updatedTestName = testSetup.testName <~ { skip: { reason: "Setup failed with: $($.error.message)" } }
    	  ---
    	  skipRun(updatedTestName, true, patternMatches)
    	}
    }
  } else skipRun(testSetup.testName, true, patternMatches)
}

/**
* Defines a new test case inside a test suite with a single assertion.
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* "It should support nested matching" in  do {
*    "foo" must beString()
* }
* ----
*/
fun in<T>(testName: String, testCases: (Null) -> MatcherResult): TestResult = in(testName, [testCases])

/**
* Defines a new test case with multiple assertions
*
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*  "It should support multiple root cases" in do {
*      var payload = {}
*      var flowVar = {a: 123}
*      ---
*     [
*         payload must beObject(),
*         flowVar must [
*              beObject(),
*              $.a must equalTo(123)
*           ]
*       ]
*  }
* ----
*/
fun in(testName: String, callback: Array<(Null) -> MatcherResult>): TestResult = do {
    var skipAll: Boolean = (dw::Runtime::prop("skipAll") default false) as Boolean
    ---
    in(testName, callback, null, skipAll)
}

@Internal(permits = [])
fun in<T>(testName: String, callback: Array<(T) -> MatcherResult>, maybeCtx: T, skipAll: Boolean): TestResult = do {
    var OK_RESULT = {name: testName, status: OK_STATUS, time: 0}

    var testToRun: String | Null = dw::Runtime::prop("testToRun")

    fun doIn(testName:String, testCases: (Null) -> MatcherResult) = do {
        //Unicode chars for helping visualizing
        var CHECK = "\u2713"
        var CROSS = "\u274C"
        var WARNING = "\u26A0"

        var testWithTime = duration(() -> dw::Runtime::try(() -> testCases(maybeCtx)))
        var timeExecution = testWithTime.time
        var testExecution: TryResult<MatcherResult> = testWithTime.result
        var result = if(testExecution.success)
            {
                name: testName,
                time: timeExecution,
                status: if(testExecution.result.matches!) OK_STATUS else FAIL_STATUS,
                errorMessage: if(testExecution.result.reasons?)
                    testExecution.result.reasons! joinBy "\n"
                else testExecution.result.description
            }
        else
            {
                name: testName,
                status: ERROR_STATUS,
                time: timeExecution,
                errorMessage: testExecution.error.kind ++  ":" ++ testExecution.error.message! ++
                if(testExecution.error.location?)
                    " at:\n" ++ testExecution.error.location!
                else "" ++
                    if(testExecution.error.stack?)
                        "\n" ++ testExecution.error.stack! reduce ($$ ++ "\n" ++ "$")
                    else
                        "\n" ++ (testExecution.error.stacktrace as String default " NO STACK ")
            }
        ---
        result
    }
    fun exec() = interceptor(
        () -> log("data-weave-testing-framework", {event: "TestStarted", name: testName, location: location(testName)}),
        () -> do {
            callback reduce
                (assertion, result = OK_RESULT) -> if (result.status == OK_STATUS) doIn(testName, assertion) else result
        },
        (result) -> log("data-weave-testing-framework", result ++ {event: "TestFinished"})
    )
    var patternMatches = testMatches(testToRun, testName)
    var skipAnnotated = testName.^skip is Object
    var result = if (skipAll or (not patternMatches) or skipAnnotated) skipRun(testName, skipAnnotated, patternMatches) else exec()
    ---
    result mergeWith { location: location(testName) }
}

/**
* Defines a new test suite with the list of test cases.
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
* ]
* ----
*/
fun describedBy(suite: String, testsToRun: Array<() -> TestResult> ): TestResult = do {
    var skipAnnotated = suite.^skip is Object
    var SKIPPED_TEST_RESULT: TestResult = {
      name: "",
      time: 0,
      status: SKIP_STATUS
    }
    fun skipRun() = interceptor(
            () -> log("data-weave-testing-framework", { event: "TestSuiteStarted", name: suite, location: location(suite) }),
            () -> {
                name: suite,
                time: 0,
                status: SKIP_STATUS,
                (skipReason: suite.^skip.reason as String) if skipAnnotated,
                tests: testsToRun map () -> SKIPPED_TEST_RESULT
            },
            (result) -> log("data-weave-testing-framework", result ++ { event: "TestSuiteFinished" })
    )
    fun exec() = interceptor(
        () -> log("data-weave-testing-framework", { event: "TestSuiteStarted", name: suite, location: location(suite) }),
        () -> do {
            var measuredResult = duration(() -> testsToRun reduce ((test, acc = []) -> acc + test()))
            var testResults = measuredResult.result
            var testExecutionTime = measuredResult.time
            ---
            {
                name: suite,
                time: testExecutionTime,
                status: OK_STATUS,
                tests: testResults
            }
        },
        (result) -> log("data-weave-testing-framework", result ++ { event: "TestSuiteFinished" })
    )
    var result = if (skipAnnotated) skipRun() else exec()
    ---
    result ++ { location: location(suite) }
}

/**
* Builds an object with all the inputs to be used as context for a specific mapping.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | dir | Directory where to look for the inputs folder and build the context from.
* |===
*
*/
fun inputsFrom(dir: String): {_?: Any} = do {
    var contextDir : String = path(path(dw::Runtime::prop("dwtestResources") default "resources", dir), "inputs")
    ---
    buildContext(contextDir) distinctBy $
}

/**
* Returns the result of reading the expected output
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name  | Description
* | dir   | Directory where to look for the out file to make assertions on your mapping test.
* |===
*
*/
fun outputFrom(dir: String) = do {
    var contextDir : String = path(dw::Runtime::prop("dwtestResources") default "resources", dir)
    var out = ls(contextDir, "out.*")[0]
    var mimeType = mimeTypeOf(out) default "text/plain"
    ---
    read(contentOf(out), mimeType)
}

/**
* Runs a specific mapping with the given context and mimetype.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | dwlFilePath   | Path inside the classpath where the runnable dataweave file is present.
* | context   | Object which contains different input values (each input name would be the key of the object).
* | mimeType   | The default mimetype if not specified on the file
* |===
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
* import * from dw::test::Asserts
* ---
* "Test MyMapping" describedBy [
*     "Assert SimpleScenario" in do {
*         evalPath("MyMapping.dwl", inputsFrom("MyMapping/SimpleScenario"), "application/json" ) must
*                   equalTo(outputFrom("MyMapping/SimpleScenario"))
*     }
*  ]
* ----
*/
fun evalPath(dwlFilePath: String, context: Object, mimeType: String) : Any = do {
    evalPath(
        { content: readUrl("classpath://" ++ dwlFilePath , "text/plain") as String, url: dwlFilePath },
        context,
        mimeType
    )
}

/**
*
* Evals a test with a given input values as a context and using the specified mimeType as default one when not specified in the file
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `testUrl` | { content: String, url: String } | The test
* | `context` | Object |  The inputs
* | `mimeType` | String | The default mimetype
* |===
*
**/
fun evalPath(testUrl: {content: String, url: String}, context: Object, mimeType: String): Any = do {
    if(mimeType == "application/java" or  mimeType == "java") do {
      //For java we directly eval it as it is not a format that can be written
        eval( testUrl.url,
                 { (testUrl.url): testUrl.content },
                 {},
                 context,
                 {
                     outputMimeType: mimeType,
                     onException: "FAIL",
                     loggerService: { log: (level, msg, ctx)-> log(msg) }
                 }).result.value
    } else do {
        var resultValue = run(
              testUrl.url,
              { (testUrl.url): testUrl.content },
              {},
              context,
              {
                          outputMimeType: mimeType,
                          onException: "FAIL",
                          loggerService: { log: (level, msg, ctx)-> log(msg) }
                       }).result.value
       ---
       read(resultValue, mimeType)
    }
}

@Internal(permits = [])
fun testMatches(testPattern: String | Null, testName: String) = testPattern match {
    case is Null -> true
    case testToRunPattern is String ->
      isBlank(testToRunPattern) or some(testToRunPattern splitBy ",", (acceptedTest) -> testName contains acceptedTest)
}

@Internal(permits = [])
fun skipRun(testName: String, skipAnnotated: Boolean, patternMatches: Boolean): TestResult = interceptor(
    //If the test was skipped due to not matching the regex on the property testToRun then we don't emit events
    //so that they don't get printed into console
    () ->
      if (patternMatches)
        log("data-weave-testing-framework", {event: "TestStarted", name: testName, location: location(testName)})
      else null,
    () -> {
        name: testName,
        time: 0,
        status: SKIP_STATUS,
        (skipReason: testName.^skip.reason as String) if skipAnnotated
    },
    (result) ->
      if (patternMatches)
        log("data-weave-testing-framework", result ++ {event: "TestFinished"})
      else null
)