%dw 2.0
import * from dw::io::file::FileSystem
import try, fail from dw::Runtime

/**
* Helper method to execute something before and after
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | before | The method to be executed before
* | value | The function to be executed
* | after | The function to be executed after
* |===
*
**/
fun interceptor<T>(before:() -> Any, value:() -> T, after: (T) -> Any): T = do {
  var beforeResult = before()
  var result = value()
  var afterResult = after(result)
  ---
  result
}

/**
* Helper function to generate input context to run mappings.
* By default, it will check for the existence of reader properties files for a given input within
* the same directory, and attempt to pass them to the `read` function within the reader properties object.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | dir    | Directory where files will be traversed and transformed into inputs for the desired mapping execution
* |===
*
**/
fun buildContext(dir: String) = do {
  var inputs = ls(dir) map (element) -> do {
    {
      ((baseNameOf(element)): kindOf(element) match {
        case "File" ->
          read(
            contentOf(element),
            mimeTypeOf(element) default "text/plain",
            if(hasReaderProperties(element))
              try(() -> read(contentOf(propertiesFileOf(element)), "text/x-java-properties")) match {
                case s if (s.success) -> s.result
                else -> {}
              }
            else
              {}
          )
        case "Folder" -> buildContext(element)
      }) if (extensionOf(element) != ".properties")
    }
  }
  ---
  { (inputs) }
}

/**
* Helper function to assert whether a properties file containing reader parameters exists
* within the same directory for a given input file
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | p | Path | Path to the input file
* |===
*
**/
fun hasReaderProperties(p: Path): Boolean = exists(p) and exists(propertiesFileOf(p))

/**
* Helper function to assemble the path to a properties file corresponding to an input
* based on implicit naming conventions.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | p | Path | Path to the input file.
* |===
*
**/
fun propertiesFileOf(p: Path): Path = do {
    var parent = parentOf(p)
    ---
    if (parent == null)
      fail("Parent of '$(p)' path is null")
    else
      path(parent as String, "$(baseNameOf(p))-config.properties")

}