package org.mule.weave.v2.module.dwb.reader

import java.util
import java.util.{ Map => JMap }
import org.mule.weave.v2.dwb.api.IWeaveValue
import org.mule.weave.v2.dwb.api.WeaveProcessor
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.reader.exceptions.DWBRuntimeExecutionException
import org.mule.weave.v2.module.dwb.reader.exceptions.InvalidProcessorClassNameException
import org.mule.weave.v2.module.pojo.ClassLoaderService
import org.mule.weave.v2.module.pojo.DefaultClassLoaderService
import org.mule.weave.v2.module.pojo.reader.ReflectionJavaValueConverter
import org.mule.weave.v2.parser.location.LocationCapable

class SimpleObjectSeqMap(seq: ObjectSeq)(implicit ctx: EvaluationContext) extends util.AbstractMap[String, IWeaveValue[_]] {
  override def size(): Int = {
    seq.size().toInt
  }

  override def isEmpty: Boolean = seq.isEmpty()

  override def containsKey(key: Any): Boolean = {
    get(key) != null
  }

  override def get(key: Any): IWeaveValue[_] = {
    if (!key.isInstanceOf[String]) {
      throw new DWBRuntimeExecutionException("Only string keys are supported in this version")
    }
    val keyStr = String.valueOf(key)
    val keyValue = KeyValue(keyStr)
    val value = seq.selectValue(keyValue)
    if (value == null) {
      null
    } else {
      WeaveValue.apply(value)
    }
  }

  override def entrySet(): util.Set[JMap.Entry[String, IWeaveValue[_]]] = {
    new EntrySet(seq)
  }

  class EntrySet(seq: ObjectSeq) extends util.AbstractSet[JMap.Entry[String, IWeaveValue[_]]] {
    override def size(): Int = seq.size().toInt

    override def iterator(): util.Iterator[JMap.Entry[String, IWeaveValue[_]]] = {
      new EntryIterator(seq.toIterator())
    }
  }

  class EntryIterator(it: Iterator[KeyValuePair]) extends util.Iterator[JMap.Entry[String, IWeaveValue[_]]] {
    override def hasNext: Boolean = it.hasNext

    override def next(): JMap.Entry[String, IWeaveValue[_]] = {
      val kvp = it.next()

      new JMap.Entry[String, IWeaveValue[_]] {
        override def getKey: String = StringType.coerce(kvp._1).evaluate.toString //TODO: support namespaces and attributes

        override def getValue: IWeaveValue[_] = WeaveValue(kvp._2)

        override def setValue(value: IWeaveValue[_]): IWeaveValue[_] = throw new UnsupportedOperationException()
      }
    }
  }

}

class RedefinedObjectSeqMap(value: ObjectSeq, processorClass: String, schema: JMap[String, IWeaveValue[_]], locationCapable: LocationCapable)(implicit _ctx: EvaluationContext) extends SimpleObjectSeqMap(value) {
  private val retriever = new RedefinedValueRetriever(value, processorClass, schema, locationCapable)

  override def get(key: Any): IWeaveValue[_] = {
    if (!key.isInstanceOf[String]) {
      throw new DWBRuntimeExecutionException("Only string keys are supported in this version")
    }
    val keyStr = String.valueOf(key)
    val keyValue = KeyValue(keyStr)
    val v = value.selectValue(keyValue)
    if (v == null) {
      val redefined = retriever.getRedefinedValue(keyStr)
      WeaveValue(redefined)
    } else {
      WeaveValue(v)
    }
  }
}

class RedefinedValueRetriever(value: ObjectSeq, processorClass: String, schema: JMap[String, IWeaveValue[_]], locationCapable: LocationCapable)(implicit ctx: EvaluationContext) {

  def getRedefinedValue(keyStr: String): Value[_] = {
    //we made sure it is redefined previously by checking the schema properties
    val weaveProcessor = getWeaveProcessor()
    val blob = BinaryType.coerce(value.selectValue(KeyValue("*"))).evaluate
    val result = weaveProcessor.get(blob, keyStr, schema)
    ReflectionJavaValueConverter.convert(result, () => locationCapable.toString)
  }

  protected def getWeaveProcessor(): WeaveProcessor = {
    val maybeClazz = ctx.serviceManager.serviceProvider.lookupCustomService(classOf[ClassLoaderService]).getOrElse(DefaultClassLoaderService).loadClass(processorClass)
    maybeClazz match {
      case Some(clazz) => clazz.newInstance().asInstanceOf[WeaveProcessor]
      case None        => throw new InvalidProcessorClassNameException(processorClass, locationCapable.location())
    }

  }
}