package org.mule.weave.v2.module.dwb.reader

import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.EmptyLocationCapable
import org.mule.weave.v2.model.structure.NameValuePair
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.LocalDateType
import org.mule.weave.v2.model.types.LocalTimeType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.TimeType
import org.mule.weave.v2.model.types.TimeZoneType
import org.mule.weave.v2.model.values.AnyValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.DwTokenType
import org.mule.weave.v2.module.dwb.reader.memory.InMemoryWeaveBinaryParser
import org.mule.weave.v2.module.dwb.writer.WeaveBinaryWriter

import java.io.InputStream
import java.math.BigInteger
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.OffsetTime
import java.time.ZoneId
import java.time.ZonedDateTime
import javax.xml.namespace.QName

class DefaultWeaveStreamReader(is: InputStream) {
  private implicit val ctx: EvaluationContext = EvaluationContext()

  private lazy val ss = SeekableStream(is)
  private lazy val reader = initReader(ss)

  private var currentEvent: Int = -2 //-2 initial pos, -1 end of stream
  private var nextEvent: Int = -2
  private var _next: Option[Value[_]] = None
  private var _current: Option[Value[_]] = None

  /**
    * Returns true if there are more parsing events and false if there are no more events
    */
  def hasNext(): Boolean = {
    if (_next.isEmpty) {
      _next = parseNext()
    }
    _next.isDefined
  }

  private def processDeclarations(tokenType: Int): Int = {
    tokenType match {
      case DwTokenType.DeclareNS =>
        reader.readNSDeclaration()
        processDeclarations(reader.readTokenType())

      case DwTokenType.DeclareName =>
        reader.readNameDeclaration()
        processDeclarations(reader.readTokenType())

      case _ =>
        tokenType
    }

  }

  private def parseNext(): Option[Value[_]] = {
    val tokenTypeWithFlags = processDeclarations(reader.readTokenType())
    if (tokenTypeWithFlags == -1) {
      return None //reached end-of-stream
    }

    val tokenType = tokenTypeWithFlags & DwTokenHelper.TOKEN_TYPE_MASK
    val value = tokenType match {
      case DwTokenType.ObjectStart =>
        NullValue //dummy value to mark that there's an object (won't be used)

      case DwTokenType.ArrayStart =>
        NullValue //dummy value to mark that there's an array (won't be used)

      case DwTokenType.StructureEnd =>
        if (DwTokenHelper.hasSchemaProps(tokenTypeWithFlags)) {
          val schema = reader.readSchema()
          new SchemaAwareDummyValue(schema)
        } else {
          NullValue
        }
      //dummy value to mark that there's an end structure (won't be used)

      case _ =>
        reader.readValue(tokenTypeWithFlags)

    }
    nextEvent = tokenTypeWithFlags
    Some(value)
  }

  /**
    * Returns an integer code that indicates the type
    * of the event the cursor is pointing to.
    */
  def getEventType: Int = {
    currentEvent
  }

  /**
    * Advances the reading cursor and returns the next token type.
    * See DwTokenType for the available types.
    */
  def next(): Int = {
    if (hasNext()) {
      currentEvent = nextEvent
      _current = _next
      _next = None
      nextEvent = -2
      currentEvent
    } else {
      -1
    }
  }

  def getQName: QName = {
    val qname = KeyType.coerce(_current.get).evaluate
    val ns = qname.namespace.get
    new QName(ns.uri, qname.name, ns.prefix)
  }

  def getLocalName: String = {
    KeyType.coerce(_current.get).evaluate.name
  }

  def getPrefix: String = {
    val key = KeyType.coerce(_current.get).evaluate
    val ns = key.namespace.get
    ns.prefix
  }

  def getNamespaceURI: String = {
    val qname = KeyType.coerce(_current.get).evaluate
    val ns = qname.namespace.get
    ns.uri
  }

  def getAttributeCount: Int = {
    getAttributes() match {
      case Some(attrs) => {
        attrs.evaluate.size()
      }
      case None => 0
    }
  }

  private def getAttributes() = {
    KeyType.coerce(_current.get).attributes
  }

  private def getAttributePair(index: Int): NameValuePair = {
    getAttributes() match {
      case Some(attrs) =>
        val nameValuePair = attrs.evaluate.apply(index).get
        nameValuePair

      case None =>
        throw new IndexOutOfBoundsException("Invalid attribute index: " + index)
    }
  }

  private def getAttributeQualifiedName(index: Int): QualifiedName = {
    getAttributePair(index)._1.evaluate
  }

  def getAttributeQName(index: Int): QName = {
    getAttributeQualifiedName(index).toQName()
  }

  def getAttributeLocalName(index: Int): String = {
    getAttributeQualifiedName(index).name
  }

  def getAttributePrefix(index: Int): String = {
    getAttributeQualifiedName(index).toQName().getPrefix
  }

  def getAttributeNamespace(index: Int): String = {
    getAttributeQualifiedName(index).toQName().getNamespaceURI
  }

  def getAttributeValueType(index: Int): Int = {
    val value = getAttributeValue(index)
    WeaveBinaryWriter.valueToTokenType(value)
  }

  def getAttributeString(index: Int): String = {
    val value = getAttributeValue(index)
    StringType.coerce(value).evaluate.toString
  }

  private def getAttributeValue(index: Int) = {
    getAttributePair(index)._2
  }

  def getAttributeInt(index: Int): Int = {
    val value = getAttributeValue(index)
    NumberType.coerce(value).evaluate.toInt
  }

  def getSchemaPropertyCount: Int = {
    getSchema().properties().size
  }

  private def getSchema(): Schema = {
    val value = _current.get
    value.schema.get
  }
  def getSchemaPropertyName(index: Int): String = {
    getSchemaProperty(index).name.evaluate
  }

  private def getSchemaProperty(index: Int) = {
    getSchema().properties().apply(index)
  }

  def getSchemaPropertyValueType(index: Int): Int = {
    val value = getSchemaProperty(index).value
    WeaveBinaryWriter.valueToTokenType(value)
  }

  def getSchemaPropertyString(index: Int): String = {
    val value = getSchemaProperty(index).value
    StringType.coerce(value).evaluate.toString
  }

  def getSchemaPropertyInt(index: Int): Int = {
    val value = getSchemaProperty(index).value
    NumberType.coerce(value).evaluate.toInt
  }
  def getString: String = {
    StringType.coerce(_current.get).evaluate.toString
  }

  private def getNumber(): org.mule.weave.v2.model.values.math.Number = {
    NumberType.coerce(_current.get).evaluate
  }

  def getInt: Int = {
    getNumber().toInt
  }

  def getLong: Long = {
    getNumber().toLong
  }

  def getBigInt: BigInteger = {
    getNumber().toBigInt.bigInteger
  }

  def getDouble: Double = {
    getNumber().toDouble
  }

  def getBigDecimal: java.math.BigDecimal = {
    getNumber().toBigDecimal.bigDecimal
  }

  def getDateTime: ZonedDateTime = {
    DateTimeType.coerce(_current.get).evaluate
  }

  def getLocalDate: LocalDate = {
    LocalDateType.coerce(_current.get).evaluate
  }

  def getLocalDateTime: LocalDateTime = {
    LocalDateTimeType.coerce(_current.get).evaluate

  }

  def getTime: OffsetTime = {
    TimeType.coerce(_current.get).evaluate
  }

  def getLocalTime: LocalTime = {
    LocalTimeType.coerce(_current.get).evaluate
  }

  def getTimeZone: ZoneId = {
    TimeZoneType.coerce(_current.get).evaluate
  }

  def getBinary: InputStream = {
    BinaryType.coerce(_current.get).evaluate
  }

  private def initReader(ss: SeekableStream): InMemoryWeaveBinaryParser = {
    val reader = new InMemoryWeaveBinaryParser("payload", ss)
    reader.readHeader()
    reader
  }

  def close(): Unit = {
    ctx.close()
    ss.close()
  }

}

class SchemaAwareDummyValue(_schema: Schema) extends AnyValue with EmptyLocationCapable {

  override def evaluate(implicit ctx: EvaluationContext): Any = throw new IllegalStateException("Shouldn't evaluate a dummy value")

  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    Some(_schema)
  }
}