package org.mule.weave.v2.module.dwb.reader.indexed

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.AlreadyMaterializedObjectValue
import org.mule.weave.v2.module.dwb.BinaryLocation
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.reader.WeaveValue
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.LocationCapable

class WeaveBinaryObject(startTokenIndex: Long, firstLcIndex: Long, lastLcIndex: Long, input: BinaryParserInput, startToken: Array[Long]) extends AlreadyMaterializedObjectValue with LocationCapable {
  private val offset = DwTokenHelper.getOffset(startToken)
  private val length = DwTokenHelper.getValueLength(startToken)
  private val depth = DwTokenHelper.getDepth(startToken)
  private var _schema: Option[Schema] = _

  override def location(): Location = {
    // start token has the ending offset, since it's needed for reading the schema
    // so we use the offset of the first key
    new BinaryLocation(DwTokenHelper.getOffset(startToken))
  }

  override def evaluate(implicit ctx: EvaluationContext): T = {
    if (firstLcIndex == -1) {
      ObjectSeq.empty
    } else {
      val depth = DwTokenHelper.getDepth(startToken)
      val hasSchema = DwTokenHelper.hasSchemaProps(startToken)
      if (!hasSchema) {
        new WeaveBinaryObjectSeq(depth, firstLcIndex, lastLcIndex, input)
      } else {
        getProcessor() match {
          case Some(processor) =>
            val schemaMap = WeaveValue.toWeaveValueMap(schema.get)
            new RedefinedWeaveBinaryObjectSeq(depth, firstLcIndex, lastLcIndex, input, processor, schemaMap, this, ctx)

          case None =>
            new WeaveBinaryObjectSeq(depth, firstLcIndex, lastLcIndex, input)
        }
      }
    }
  }

  private def getProcessor()(implicit ctx: EvaluationContext): Option[String] = {
    for (
      theSchema <- schema;
      processor <- theSchema.valueOf("processor")
    ) yield {
      StringType.coerce(processor).evaluate.toString
    }
  }

  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    import org.mule.weave.v2.module.core.xml.reader.indexed.TokenHelpers._
    if (_schema == null) {
      if (DwTokenHelper.hasSchemaProps(startToken)) {
        val schemaOffset = offset + length
        val lasChildLcEntry = input.locationCaches.apply(depth + 1).apply(lastLcIndex)
        val lastChildTokenIndex = lasChildLcEntry.getTokenIndex
        // Schema properties are at the same level as the object token.
        // Start searching for this after the last child key token
        val schemaPropKeyIndex = input.tokenArray.indexWhere(
          (x) => DwTokenHelper.getDepth(x) == depth,
          lastChildTokenIndex)
        val schema = WeaveBinaryValue.readSchema(input, schemaPropKeyIndex.toInt - 1, schemaOffset)
        _schema = Some(schema)
      } else {
        _schema = None
      }
    }
    _schema
  }

}
