package org.mule.weave.v2.module.dwb.reader.indexed

import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.DwTokenType
import org.mule.weave.v2.module.dwb.reader.exceptions.DWBRuntimeExecutionException

import scala.collection.mutable.ArrayBuffer

object WeaveBinaryValue {
  def apply(tokenIndex: Long, lcIndexMaybe: Option[Long], input: BinaryParserInput): Value[_] = {
    val token = input.tokenArray(tokenIndex)
    val tokenType = DwTokenHelper.getTokenType(token)

    tokenType match {
      case DwTokenType.ObjectStart   => createDwBinaryObject(tokenIndex, input, lcIndexMaybe, token)
      case DwTokenType.ArrayStart    => createDwBinaryArray(tokenIndex, input, lcIndexMaybe, token)
      case DwTokenType.String8       => new WeaveBinaryString(tokenIndex, token, input)
      case DwTokenType.String32      => new WeaveBinaryString(tokenIndex, token, input)

      case DwTokenType.Int           => new WeaveBinaryInt(token, input)
      case DwTokenType.Long          => new WeaveBinaryLong(token, input)
      case DwTokenType.Double        => new WeaveBinaryDouble(token, input)
      case DwTokenType.BigInt        => new WeaveBinaryBigInt(token, input)
      case DwTokenType.BigDecimal    => new WeaveBinaryBigDecimal(token, input)

      case DwTokenType.True          => BooleanValue.TRUE_BOOL
      case DwTokenType.False         => BooleanValue.FALSE_BOOL

      case DwTokenType.Null          => NullValue

      case DwTokenType.DateTime      => new WeaveBinaryDateTime(token, input)
      case DwTokenType.LocalDate     => new WeaveBinaryLocalDate(token, input)
      case DwTokenType.Period        => new WeaveBinaryPeriod(token, input)
      case DwTokenType.LocalDateTime => new WeaveBinaryLocalDateTime(token, input)
      case DwTokenType.Time          => new WeaveBinaryTime(token, input)
      case DwTokenType.LocalTime     => new WeaveBinaryLocalTime(token, input)
      case DwTokenType.TimeZone      => new WeaveBinaryTimeZone(token, input)
      case DwTokenType.Binary        => new WeaveBinaryBlob(tokenIndex, token, input)

      case DwTokenType.Regex         => new WeaveBinaryRegex(token, input)
      case DwTokenType.Range         => new WeaveBinaryRange(token, input)
      case _ =>
        throw new DWBRuntimeExecutionException("Got '" + DwTokenType.getNameFor(tokenType) + "' token while reading values")

    }
  }

  private def getFirstAndLastLcIndex(startTokenIndex: Long, lcIndexMaybe: Option[Long], input: BinaryParserInput): (Long, Long) = {
    val locationCaches = input.locationCaches
    val depth = DwTokenHelper.getDepth(input.tokenArray(startTokenIndex))
    val lcIndex = lcIndexMaybe.getOrElse(locationCaches.indexOf(startTokenIndex, depth))
    locationCaches.getFirstAndLastChildren(lcIndex, depth)
  }

  private def createDwBinaryObject(index: Long, input: BinaryParserInput, lcIndexMaybe: Option[Long] = None, startToken: Array[Long]): Value[_] = {
    val (firstLcIndex, lastLcIndex) = getFirstAndLastLcIndex(index, lcIndexMaybe, input)
    new WeaveBinaryObject(index, firstLcIndex, lastLcIndex, input, startToken)
  }

  private def createDwBinaryArray(index: Long, input: BinaryParserInput, lcIndexMaybe: Option[Long] = None, startToken: Array[Long]): Value[_] = {
    val (firstLcIndex, lastLcIndex) = getFirstAndLastLcIndex(index, lcIndexMaybe, input)
    new WeaveBinaryArray(index, firstLcIndex, lastLcIndex, input, startToken)
  }

  def readLocalName(input: BinaryParserInput, keyToken: Array[Long]): String = {
    val nameIndex = DwTokenHelper.getNameIndex(keyToken)
    val name = input.names(nameIndex)
    name
  }

  def readSchema(input: BinaryParserInput, valueTokenIndex: Long, offset: Long): Schema = {
    input.seekableStream.seek(offset)
    val tokenArray = input.tokenArray
    val props = new ArrayBuffer[SchemaProperty]
    val dis = input.dataInputStream
    val propCount = dis.readUnsignedShort()

    var i = 0
    while (i < propCount) {
      val pairPos = valueTokenIndex + (i * 2)
      val keyStrValue = StringValue(readLocalName(input, tokenArray(pairPos + 1)))
      val value = WeaveBinaryValue.apply(pairPos + 2, None, input)
      props += SchemaProperty(keyStrValue, value)
      i += 1
    }
    Schema(props)
  }
}