package org.mule.weave.v2.module.dwb.writer

import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.math
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.NameSeq
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.BooleanType.T
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.LocalDateType
import org.mule.weave.v2.model.types.LocalTimeType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.PeriodType
import org.mule.weave.v2.model.types.RangeType
import org.mule.weave.v2.model.types.RegexType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.TimeType
import org.mule.weave.v2.model.types.TimeZoneType
import org.mule.weave.v2.model.values.DWRange
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.core.common.LocationCacheBuilder
import org.mule.weave.v2.module.core.xml.reader.indexed.TokenArray
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.dwb.DefaultWeaveBinaryDataFormat
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.DwTokenHelper.NO_NAMESPACE
import org.mule.weave.v2.module.dwb.DwTokenType
import org.mule.weave.v2.module.dwb.DwTokenType.DwTokenType
import org.mule.weave.v2.module.dwb.WeaveBinaryDataFormat
import org.mule.weave.v2.module.dwb.WeaveBinaryUtils
import org.mule.weave.v2.module.dwb.WeaveKeyToken
import org.mule.weave.v2.module.dwb.WeaveValueToken
import org.mule.weave.v2.module.dwb.reader.NumberPrecisionHelper
import org.mule.weave.v2.module.dwb.reader.exceptions.DWBRuntimeExecutionException
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.writer.ConfigurableBufferSize
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.location.LocationCapable
import org.mule.weave.v2.parser.location.UnknownLocation

import java.io.BufferedOutputStream
import java.io.InputStream
import java.io.OutputStream
import java.math.BigInteger
import java.nio.charset.Charset
import java.time.ZoneOffset

import scala.collection.mutable

class WeaveBinaryWriter(os: OutputStream, val settings: WeaveBinaryWriterSettings = DefaultWeaveBinaryDataFormat.createWriterSettings())(implicit ctx: EvaluationContext) extends Writer {

  import WeaveBinaryWriter._

  private val charset = Charset.forName("UTF-8")

  private val output = new LongCountDataOutputStream(new BufferedOutputStream(os))
  private val tokenBuffer = ctx.registerCloseable(new TokenArray())
  private val lcBuilder = new LocationCacheBuilder(DwTokenHelper)

  private val namespaces = new IndexHashSet[Namespace]()
  private val names = new IndexHashSet[String]()
  private var depth = 0

  override def dataFormat: Option[DataFormat[_, _]] = Some(new WeaveBinaryDataFormat)

  override def result: Any = {
    output.flush()
    os
  }

  override def flush(): Unit = {
    output.flush()
  }

  override def close(): Unit = {
    output.close()
  }

  private def writeByte(byte: Int): Unit = {
    //    if (byte > 255 || byte < 0) {
    //      throw new IllegalArgumentException("Value: '" + byte + "' can't be written as a byte (0 to 255)")
    //    }
    //TODO: check if this bounds checking is expensive
    output.writeByte(byte)
  }

  /**
    * Returns the index of the namespace, declaring it if necessary
    */
  private def getOrDeclareNS(ns: Namespace): Int = {
    val maybeIndex = namespaces.indexOf(ns)
    if (maybeIndex != -1) {
      maybeIndex
    } else {
      val newIndex = namespaces.put(ns)
      if (!settings.writeIndex) {
        writeTokenType(DwTokenType.DeclareNS, hasSchema = false)
        output.writeUTF(ns.prefix + ":" + ns.uri)
      }
      newIndex
    }
  }

  /**
    * Returns the index of the name, declaring it if necessary
    * Note: The index is a short
    */
  private def getOrDeclareName(name: String) = {
    val maybeIndex = names.indexOf(name)
    if (maybeIndex != -1) {
      maybeIndex
    } else {
      val newIndex = names.put(name)
      if (!settings.writeIndex) {
        writeTokenType(DwTokenType.DeclareName, hasSchema = false)
        output.writeUTF(name)
      }
      newIndex
    }
  }

  private def writeLocalName(nameIndex: Int, tokenType: DwTokenType): Unit = {
    if (!settings.writeIndex) {
      writeTokenType(tokenType, hasSchema = false)
      output.writeShort(nameIndex)
    }
  }

  private def writeNS(nsIndex: Int): Unit = {
    if (!settings.writeIndex) {
      output.writeShort(nsIndex)
    }
  }

  /**
    * The place this function is invoked in matters because it depends on `output.size` and `depth` field values
    */
  private def createKeyToken(name: String, nameIndex: Int, nsIndex: Int, tokenType: DwTokenType): Array[Long] = {
    val length = WeaveBinaryUtils.getUTFByteLength(name)
    val nameHash = DwTokenHelper.hash(name)
    WeaveKeyToken(output.size(), tokenType, depth, length, nameHash, nameIndex, nsIndex)
  }

  /**
    * The place this function is invoked in matters because it depends on `output.size` and `depth` field values
    */
  private def createValueToken(length: Long, tokenType: DwTokenType, hasSchema: Boolean): Array[Long] = {
    WeaveValueToken(output.size(), tokenType, depth, length, hasSchema)
  }

  def writeKey(key: KeyType.V)(implicit ctx: EvaluationContext): Unit = {
    val qname = key.evaluate
    val attrsMaybe = key.attributes
    val keyTokenType = getKeyTokenType(qname.namespace.isDefined, attrsMaybe.isDefined)
    writeQName(qname, keyTokenType, putTokenInLC = true)
    writeAttributes(attrsMaybe)
  }

  def writeQName(qname: QualifiedName, keyTokenType: DwTokenType, putTokenInLC: Boolean): Unit = {
    val maybeNS = qname.namespace
    val name = qname.name
    val nameIndex = getOrDeclareName(name)

    val t = if (maybeNS.isDefined) {
      val nsIndex = getOrDeclareNS(maybeNS.get)
      val token = createKeyToken(name, nameIndex, nsIndex, keyTokenType)
      writeLocalName(nameIndex, keyTokenType)
      writeNS(nsIndex)
      token
    } else {
      val token = createKeyToken(name, nameIndex, NO_NAMESPACE, keyTokenType)
      writeLocalName(nameIndex, keyTokenType)
      token
    }
    addToCaches(t, putTokenInLC)
  }

  def writeAttributes(attrsMaybe: Option[Value[NameSeq]])(implicit ctx: EvaluationContext): Unit = {
    if (attrsMaybe.isDefined) {
      val attrs = attrsMaybe.get
      val nameSeq = attrs.evaluate.toStream()
      writeShort(nameSeq.size)
      for (nameValuePair <- nameSeq) {
        val attrName = nameValuePair._1.evaluate
        val keyTokenType = getKeyTokenType(attrName.namespace.isDefined, hasAttrs = false)
        writeQName(attrName, keyTokenType, putTokenInLC = false)
        doWriteValue0(nameValuePair._2, putTokenInLC = false)
      }
    }
  }

  def writeAttributeKey(qname: QualifiedName): Unit = {
    val keyTokenType = getKeyTokenType(qname.namespace.isDefined, hasAttrs = false)
    writeQName(qname, keyTokenType, putTokenInLC = false)
  }

  def writeSchemaPropertyKey(name: String): Unit = {
    writeQName(QualifiedName(name, None), DwTokenType.Key, putTokenInLC = false)
  }

  protected def writeSchemaProperty(prop: SchemaProperty)(implicit ctx: EvaluationContext): Unit = {
    val name = prop.name.evaluate
    writeSchemaPropertyKey(name)
    doWriteValue0(prop.value, putTokenInLC = false)
  }

  def writeShort(number: Int): Unit = {
    output.writeShort(number)
  }

  protected def writeSchema(schemaMaybe: Option[Schema])(implicit ctx: EvaluationContext): Unit = {
    schemaMaybe match {
      case Some(schema) =>
        val properties = schema.properties()
        writeShort(properties.size)
        for (property <- properties) {
          if (!property.internal) {
            writeSchemaProperty(property)
          }
        }
      case None =>
      //do nothing
    }
  }

  override protected def doWriteValue(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    doWriteValue0(v, putTokenInLC = true)
  }

  def doWriteValue0(v: Value[_], putTokenInLC: Boolean)(implicit ctx: EvaluationContext): Unit = {
    var schemaMaybe = v.schema
    var hasSchema = schemaMaybe.isDefined
    v match {
      case t if KeyType.accepts(t) =>
        writeKey(v.asInstanceOf[KeyType.V])

      case t if StringType.accepts(t) =>
        val str = v.evaluate.toString
        writeString(putTokenInLC, hasSchema, str)

      case t if BooleanType.accepts(t) =>
        val bool: T = v.asInstanceOf[Value[T]].evaluate
        writeBoolean(putTokenInLC, hasSchema, bool)

      case t if NumberType.accepts(t) =>
        val numberValue = v.asInstanceOf[Value[math.Number]]
        val classMaybe = schemaMaybe.flatMap(_.valueOf("class")).map(StringType.coerce(_).evaluate.toString)
        val number = numberValue.evaluate
        if (classMaybe.isDefined) {
          val s = schemaMaybe.get
          val properties = s.properties()
          if (properties.size > 1) {
            val filteredSchema = s.properties().filterNot(_.name.evaluate.equals("class"))
            schemaMaybe = Some(Schema(filteredSchema))
          } else {
            schemaMaybe = None
            hasSchema = false
          }
        }
        writeNumber(putTokenInLC, hasSchema, number, classMaybe)

      case t if NullType.accepts(t) =>
        writeNull(putTokenInLC, hasSchema)

      case t if RangeType.accepts(t) =>
        val range = RangeType.coerce(v).evaluate
        writeRange(putTokenInLC, hasSchema, range)

      case t if ArrayType.accepts(t) =>
        writeStartArray(putTokenInLC)
        val arrIterator = v.evaluate.asInstanceOf[ArraySeq].toIterator()
        while (arrIterator.hasNext) {
          val item = arrIterator.next()
          doWriteValue0(item, putTokenInLC = true)
        }
        writeEndArray(hasSchema)

      case t if ObjectType.accepts(t) =>
        writeStartObject(putTokenInLC)
        val keyValuePairs = v.evaluate.asInstanceOf[ObjectSeq].toIterator()
        while (keyValuePairs.hasNext) {
          val keyValuePair = keyValuePairs.next()
          val key = keyValuePair._1
          val value = keyValuePair._2
          writeKey(key.asInstanceOf[KeyType.V])
          doWriteValue0(value, putTokenInLC = false)
        }
        writeEndObject(hasSchema)

      case t if FunctionType.accepts(t) =>
        throw new DWBRuntimeExecutionException("Writing function values is not supported.")

      case t if RegexType.accepts(t) =>
        val str = StringType.coerce(v).evaluate.toString
        writeWithString(str, DwTokenType.Regex, putTokenInLC, hasSchema)

      case t if DateTimeType.accepts(t) =>
        val zonedDateTime = v.asInstanceOf[Value[DateTimeType.T]].evaluate
        writeDateTime(putTokenInLC, hasSchema, zonedDateTime)

      case t if LocalDateTimeType.accepts(t) =>
        val localDateTime = v.asInstanceOf[Value[LocalDateTimeType.T]].evaluate
        writeLocalDateTime(putTokenInLC, hasSchema, localDateTime)

      case t if LocalDateType.accepts(t) =>
        val localDate = v.asInstanceOf[Value[LocalDateType.T]].evaluate
        writeLocalDate(putTokenInLC, hasSchema, localDate)

      case t if TimeType.accepts(t) =>
        val time = v.asInstanceOf[Value[TimeType.T]].evaluate
        writeOffsetTime(putTokenInLC, hasSchema, time)

      case t if LocalTimeType.accepts(t) =>
        val localTime = v.asInstanceOf[Value[LocalTimeType.T]].evaluate
        writeLocalTime(putTokenInLC, hasSchema, localTime)

      case t if TimeZoneType.accepts(t) =>
        val timezone = v.asInstanceOf[Value[TimeZoneType.T]].evaluate
        writeTimeZone(putTokenInLC, hasSchema, timezone)

      case t if BinaryType.accepts(t) =>
        val binary = v.asInstanceOf[Value[BinaryType.T]].evaluate
        writeBinary(putTokenInLC, hasSchema, binary)

      case t if PeriodType.accepts(t) =>
        val str: String = StringType.coerce(v).evaluate.toString
        writeWithString(str, DwTokenType.Period, putTokenInLC, hasSchema)

      case t =>
        throw new DWBRuntimeExecutionException("Got unexpected type '" + t.toString + "' while writing")

    }
    writeSchema(schemaMaybe)
  }

  def writeRange(putTokenInLC: T, hasSchema: T, range: DWRange): Unit = {
    writeTokenType(DwTokenType.Range, hasSchema)
    val token = createValueToken(8, DwTokenType.Range, hasSchema)
    addToCaches(token, putTokenInLC)
    output.writeInt(range.start.toInt)
    output.writeInt(range.end.toInt)
  }

  private def longToInt(longSize: Long): Int = {
    if (longSize.isValidInt) {
      longSize.toInt
    } else {
      throw new IllegalArgumentException("The binary file you are trying to write is too big. Size: " + longSize)
    }
  }

  def writeBinary(putTokenInLC: Boolean, hasSchema: Boolean, byteArray: Array[Byte]): Unit = {
    val length = byteArray.length
    if (!settings.writeIndex) {
      writeTokenType(DwTokenType.Binary, hasSchema)
      output.writeInt(longToInt(length))
    }
    val token = createValueToken(length, DwTokenType.Binary, hasSchema)
    output.write(byteArray)
    addToCaches(token, putTokenInLC)
  }

  def writeBinary(putTokenInLC: Boolean, hasSchema: Boolean, inputStream: InputStream)(implicit ctx: EvaluationContext): Unit = {
    val stream = SeekableStream(inputStream)
    val streamSize = stream.size()

    if (!settings.writeIndex) {
      writeTokenType(DwTokenType.Binary, hasSchema)
      output.writeInt(longToInt(streamSize))
    }
    val token = createValueToken(streamSize, DwTokenType.Binary, hasSchema)
    var read = stream.read()
    while (read != -1) {
      output.write(read)
      read = stream.read()
    }
    addToCaches(token, putTokenInLC)
  }

  def writeNull(putTokenInLC: T, hasSchema: T): Unit = {
    val token = createValueToken(length = 0, tokenType = DwTokenType.Null, hasSchema)
    addToCaches(token, putTokenInLC)
    writeTokenType(DwTokenType.Null, hasSchema)
  }

  def writeBoolean(putTokenInLC: T, hasSchema: T, bool: T): Unit = {
    val dwTokenType = if (bool) DwTokenType.True else DwTokenType.False
    val token = createValueToken(0, dwTokenType, hasSchema)
    addToCaches(token, putTokenInLC)
    writeTokenType(dwTokenType, hasSchema)
  }

  def writeEndArray(hasSchema: Boolean): Unit = {
    import org.mule.weave.v2.module.core.xml.reader.indexed.TokenHelpers._
    depth -= 1
    writeTokenType(DwTokenType.StructureEnd, hasSchema)
    if (hasSchema) {
      //this updates the length of the start array, which is used to know where the schema starts
      //and the new token says the array has a schema
      val lcToken = lcBuilder.lastToken(depth)
      val globalTokenIndex = lcToken.getTokenIndex
      val oldStartToken = tokenBuffer.apply(globalTokenIndex)
      val startOffset = DwTokenHelper.getOffset(oldStartToken)
      val length = output.size() - startOffset
      val newStartToken = WeaveValueToken(startOffset, DwTokenType.ArrayStart, depth, length, hasSchema)
      tokenBuffer.update(globalTokenIndex, newStartToken)
    }
  }

  def writeStartArray(putTokenInLC: Boolean): Unit = {
    val token = createValueToken(length = 0, tokenType = DwTokenType.ArrayStart, hasSchema = false)
    addToCaches(token, putTokenInLC)
    writeTokenType(DwTokenType.ArrayStart, hasSchema = false)
    depth += 1
  }

  def writeStartObject(putTokenInLC: Boolean): Unit = {
    val token = createValueToken(length = 0, tokenType = DwTokenType.ObjectStart, hasSchema = false)
    addToCaches(token, putTokenInLC)
    writeTokenType(DwTokenType.ObjectStart, hasSchema = false)
    depth += 1
  }

  def writeEndObject(hasSchema: Boolean): Unit = {
    depth -= 1
    writeTokenType(DwTokenType.StructureEnd, hasSchema)
    if (hasSchema) {
      //this updates the length of the start object, which is used to know where the schema starts
      //and the new token says the object has a schema
      val index = oldStartTokenIndex()
      val oldStartToken = tokenBuffer.apply(index)
      val startOffset = DwTokenHelper.getOffset(oldStartToken)
      val length = output.size() - startOffset
      val newStartToken = WeaveValueToken(startOffset, DwTokenType.ObjectStart, depth, length, hasSchema)
      tokenBuffer.update(index, newStartToken)
    }
  }

  private def oldStartTokenIndex(): Long = {
    import org.mule.weave.v2.module.core.xml.reader.indexed.TokenHelpers._
    val lcToken = lcBuilder.lastToken(depth)
    val globalTokenIndex = lcToken.getTokenIndex
    val oldToken = tokenBuffer(globalTokenIndex)
    if (DwTokenHelper.getTokenType(oldToken) == DwTokenType.ObjectStart) {
      globalTokenIndex //this is only true in the object as root case
    } else {
      globalTokenIndex + 1
    }
  }

  private def writeNumber(putTokenInLC: Boolean, hasSchema: Boolean, number: math.Number, classMaybe: Option[String]): Unit = {
    if (classMaybe.isDefined) {
      writeNumberWithClass(putTokenInLC, hasSchema, number, classMaybe.get)
    } else {
      writeNumberWithoutClass(putTokenInLC, hasSchema, number)
    }
  }

  private def writeNumberWithClass(putTokenInLC: Boolean, hasSchema: Boolean, number: math.Number, className: String): Unit = {
    if (NumberPrecisionHelper.isInt(className)) {
      writeInt(putTokenInLC, hasSchema, number.toInt)
    } else if (NumberPrecisionHelper.isLong(className)) {
      writeLong(putTokenInLC, hasSchema, number.toLong)
    } else if (NumberPrecisionHelper.isDouble(className)) {
      writeDouble(putTokenInLC, hasSchema, number.toDouble)
    } else if (NumberPrecisionHelper.isBigInt(className)) {
      val bigInt = number.toBigInt.bigInteger
      writeBigInt(putTokenInLC, hasSchema, bigInt)
    } else if (NumberPrecisionHelper.isBigDecimal(className)) {
      val bigDecimal = number.toBigDecimal.bigDecimal
      writeBigDecimal(putTokenInLC, hasSchema, bigDecimal)
    } else {
      val message = "Number class '" + className + "' not present in data-weave binary format"
      throw new DWBRuntimeExecutionException(message, UnknownLocation)
    }
  }

  private def writeNumberWithoutClass(putTokenInLC: Boolean, hasSchema: Boolean, number: math.Number): Unit = {
    if (number.isWhole) {
      if (number.withinInt) {
        writeInt(putTokenInLC, hasSchema, number.toInt)
      } else if (number.withinLong) {
        writeLong(putTokenInLC, hasSchema, number.toLong)
      } else {
        val bigInt = number.toBigInt.bigInteger
        writeBigInt(putTokenInLC, hasSchema, bigInt)
      }
    } else if (number.underlying().isInstanceOf[BigDecimal]) {
      val bigDecimal = number.toBigDecimal.bigDecimal
      writeBigDecimal(putTokenInLC, hasSchema, bigDecimal)
    } else {
      writeDouble(putTokenInLC, hasSchema, number.toDouble)
    }
  }

  def writeBigDecimal(putTokenInLC: Boolean, hasSchema: Boolean, bigDecimal: java.math.BigDecimal): Unit = {
    val str = bigDecimal.toString
    writeWithString(str, DwTokenType.BigDecimal, putTokenInLC, hasSchema)
  }

  private def writeWithString(str: String, tokenType: DwTokenType.DwTokenType, putTokenInLC: Boolean, hasSchema: Boolean): Unit = {
    writeTokenType(tokenType, hasSchema)
    val strBytes = str.getBytes(charset)
    val length = strBytes.length
    output.writeShort(length)
    val token = createValueToken(length, tokenType, hasSchema)
    addToCaches(token, putTokenInLC)
    output.write(strBytes)
  }

  def writeDouble(putTokenInLC: Boolean, hasSchema: Boolean, d: Double): Unit = {
    writeTokenType(DwTokenType.Double, hasSchema)
    val token = createValueToken(length = 8, DwTokenType.Double, hasSchema)
    addToCaches(token, putTokenInLC)
    output.writeDouble(d)
  }

  def writeBigInt(putTokenInLC: Boolean, hasSchema: Boolean, bigInt: BigInteger): Unit = {
    writeTokenType(DwTokenType.BigInt, hasSchema)
    val bigIntBytes = bigInt.toByteArray
    val length = bigIntBytes.length
    output.writeShort(length)
    val token = createValueToken(length, DwTokenType.BigInt, hasSchema)
    addToCaches(token, putTokenInLC)
    output.write(bigIntBytes)
  }

  def writeLong(putTokenInLC: Boolean, hasSchema: Boolean, l: Long): Unit = {
    writeTokenType(DwTokenType.Long, hasSchema)
    val token = createValueToken(length = 8, DwTokenType.Long, hasSchema)
    addToCaches(token, putTokenInLC)
    output.writeLong(l)
  }

  def writeInt(putTokenInLC: Boolean, hasSchema: Boolean, number: Int): Unit = {
    writeTokenType(DwTokenType.Int, hasSchema)
    val token = createValueToken(length = 4, DwTokenType.Int, hasSchema)
    addToCaches(token, putTokenInLC)
    output.writeInt(number)
  }

  def writeTimeZone(putTokenInLC: Boolean, hasSchema: Boolean, timezone: TimeZoneType.T): Unit = {
    val zoneId = timezone.getId
    val zoneIdIndex = getOrDeclareName(zoneId)
    writeTokenType(DwTokenType.TimeZone, hasSchema)
    val token = createValueToken(length = 2, tokenType = DwTokenType.TimeZone, hasSchema)
    output.writeShort(zoneIdIndex)
    addToCaches(token, putTokenInLC)
  }

  def writeLocalTime(putTokenInLC: Boolean, hasSchema: Boolean, localTime: LocalTimeType.T): Unit = {
    val nanoOfDay = localTime.toNanoOfDay
    writeTokenType(DwTokenType.LocalTime, hasSchema)
    val token = createValueToken(length = 8, tokenType = DwTokenType.LocalTime, hasSchema)
    output.writeLong(nanoOfDay)
    addToCaches(token, putTokenInLC)
  }

  def writeOffsetTime(putTokenInLC: Boolean, hasSchema: Boolean, time: TimeType.T): Unit = {
    val nanoOfDay = time.toLocalTime.toNanoOfDay
    val offsetId = time.getOffset.getId
    val offsetIdIndex = getOrDeclareName(offsetId)
    writeTokenType(DwTokenType.Time, hasSchema)
    val token = createValueToken(length = 10, tokenType = DwTokenType.Time, hasSchema)
    output.writeLong(nanoOfDay)
    output.writeShort(offsetIdIndex)
    addToCaches(token, putTokenInLC)
  }

  def writeLocalDateTime(putTokenInLC: Boolean, hasSchema: Boolean, localDateTime: LocalDateTimeType.T): Unit = {
    val unixTimestamp = localDateTime.toEpochSecond(ZoneOffset.UTC)
    val nanos = localDateTime.getNano
    writeTokenType(DwTokenType.LocalDateTime, hasSchema)
    val token = createValueToken(length = 12, tokenType = DwTokenType.LocalDateTime, hasSchema)
    output.writeLong(unixTimestamp)
    output.writeInt(nanos)
    addToCaches(token, putTokenInLC)
  }

  def writeLocalDate(putTokenInLC: Boolean, hasSchema: Boolean, localDate: LocalDateType.T): Unit = {
    val epochDay = localDate.toEpochDay
    writeTokenType(DwTokenType.LocalDate, hasSchema)
    val token = createValueToken(length = 8, tokenType = DwTokenType.LocalDate, hasSchema)
    output.writeLong(epochDay)
    addToCaches(token, putTokenInLC)
  }

  def writeDateTime(putTokenInLC: Boolean, hasSchema: Boolean, zonedDateTime: DateTimeType.T): Unit = {
    val unixTimestamp = zonedDateTime.toEpochSecond
    val nanos = zonedDateTime.getNano
    val zoneIdStr = zonedDateTime.getZone.getId
    val zoneIndex = getOrDeclareName(zoneIdStr)
    writeTokenType(DwTokenType.DateTime, hasSchema)
    val token = createValueToken(length = 14, tokenType = DwTokenType.DateTime, hasSchema)
    output.writeLong(unixTimestamp)
    output.writeInt(nanos)
    output.writeShort(zoneIndex)
    addToCaches(token, putTokenInLC)
  }

  def writeString(putTokenInLC: Boolean, hasSchema: Boolean, str: String): Unit = {
    //TODO: Add optimized case for smaller strings, which doesn't use an additional byte for the length
    val strBytes = str.getBytes(charset)
    val length = strBytes.length
    val tokenType = getStringTokenType(length)

    if (!settings.writeIndex) {
      writeTokenType(tokenType, hasSchema)
      if (length < 256) {
        output.writeByte(length)
      } else {
        output.writeInt(length)
      }
    }
    val token = createValueToken(length, tokenType, hasSchema)
    output.write(strBytes)
    addToCaches(token, putTokenInLC)
  }

  def writeTokenType(tokenType: DwTokenType, hasSchema: Boolean): Unit = {
    if (!settings.writeIndex) {
      val combinedTokenType = {
        val hasSchemaProp = booleanToInt(hasSchema) << DwTokenHelper.SCHEMA_FLAG_RIGHT_BITS_BYTE
        hasSchemaProp | tokenType
      }
      writeByte(combinedTokenType)
    }
  }

  override def startDocument(location: LocationCapable): Unit = {
    writeMagicWord()
    writeVersion()
    writeIndexPresence()
  }

  /**
    * Magic word used to identify the file, similar to file extensions.
    * (4 bytes)
    */
  private def writeMagicWord(): Unit = {
    output.writeInt(WeaveBinaryWriter.MAGIC_WORD)
  }

  /**
    * (1 byte)
    */
  private def writeVersion(): Unit = {
    output.write(1)
  }

  /**
    * (1 byte)
    */
  private def writeIndexPresence(): Unit = {
    output.write(booleanToInt(settings.writeIndex))
  }

  override def doEndDocument(location: LocationCapable): Unit = {
    if (settings.writeIndex) {
      writeIndex()
    }
    super.doEndDocument(location)
  }

  private def toInt(n: Long) = {
    if (n <= Integer.MAX_VALUE) {
      n.toInt
    } else {
      throw new RuntimeException("Number " + n + " is too big to fit in an Integer")
    }
  }

  /**
    * Writes the following (at the end):
    * |
    * |-> Name Entry          (N * (2 bytes + NAME_LENGTH))
    * |-> Namespace Entry     (N * (2 bytes + NAMESPACE_LENGTH))
    * |-> Global Tokens Entry (N * 16 bytes)
    * |
    * |-> LocationCaches length (2 bytes)
    * |---> LC Level length     (8 bytes)
    * |------> LC Entry         (N * 16 bytes)
    * |
    * |-> names      index bytes (4 bytes)
    * |-> names      index count (4 bytes)
    * |-> namespaces index bytes (4 bytes)
    * |-> namespaces index count (4 bytes)
    * |-> Global     index length in longs (8 bytes)
    * |-> LC         index length in bytes (8 bytes)
    */
  private def writeIndex(): Unit = {
    //names
    var initialPosition = output.size()
    var nameCount = 0 //we need the count since strings have variable length
    for (name <- names.iterator) {
      val strBytes = name.getBytes(charset)
      val length = strBytes.length
      output.writeShort(length)
      output.write(strBytes)
      nameCount += 1
    }
    val nameBytes = toInt(output.size() - initialPosition)

    //namespaces
    initialPosition = output.size()
    var nsCount = 0
    for (ns <- namespaces.iterator) {
      output.writeUTF(ns.prefix + ":" + ns.uri)
      nsCount += 1
    }
    val nsBytes = toInt(output.size() - initialPosition)

    val iterator = tokenBuffer.longIterator()
    while (iterator.hasNext) {
      //TODO: optimize writing this token buffer
      val halfToken = iterator.nextLong()
      output.writeLong(halfToken) //Global Tokens Entry (N * 16 bytes)
    }

    //global tokens
    initialPosition = output.size()
    val locationCaches = ctx.registerCloseable(lcBuilder.build())
    output.writeShort(locationCaches.size) //LocationCaches length (2 bytes)

    //location caches
    for (lc <- locationCaches.iterator) {
      output.writeLong(lc.size) //LC Level length (8 bytes)
      val longIterator = lc.iterator()
      while (longIterator.hasNext) {
        output.writeLong(longIterator.nextLong()) //LC Entry (M * 16 bytes)
      }
    }

    val lcIndexLength = output.size() - initialPosition
    output.writeInt(nameBytes)
    output.writeInt(nameCount)
    output.writeInt(nsBytes)
    output.writeInt(nsCount)
    output.writeLong(tokenBuffer.longsCount) //Global index length (8 bytes)
    output.writeLong(lcIndexLength) //LC     index length in bytes (8 bytes)
  }

  private def addToCaches(valueToken: Array[Long], putTokenInLC: Boolean): Unit = {
    addToken(valueToken)
    if (putTokenInLC) {
      addLcToken(valueToken)
    }
  }

  private def addToken(token: Array[Long]): Unit = {
    tokenBuffer += token
  }

  private def addLcToken(token: Array[Long]): Unit = {
    val tokenIndex = tokenBuffer.length - 1
    lcBuilder.addToken(token, tokenIndex)
  }

}

object WeaveBinaryWriter {
  val VERSION: Int = 1
  val HEADER_BYTES: Int = 6
  val MAGIC_WORD: Int = 0x7F414C46 // 0x7F 'A' 'L' 'F'

  private val charset = Charset.forName("UTF-8")

  def apply(targetProvider: TargetProvider, settings: WeaveBinaryWriterSettings)(implicit ctx: EvaluationContext): WeaveBinaryWriter = {
    new WeaveBinaryWriter(targetProvider.asOutputStream, settings)
  }

  def booleanToInt(boolean: Boolean): Byte = {
    if (boolean) 1 else 0
  }

  def getStringTokenType(length: Int): Int = {
    if (length < 256) {
      DwTokenType.String8
    } else {
      DwTokenType.String32
    }
  }

  def getKeyTokenType(hasNs: Boolean, hasAttrs: Boolean): DwTokenType = {
    if (hasNs && hasAttrs) {
      DwTokenType.KeyWithNSAttr
    } else if (hasNs) {
      DwTokenType.KeyWithNS
    } else if (hasAttrs) {
      DwTokenType.KeyWithAttr
    } else {
      DwTokenType.Key
    }
  }

  def valueToTokenType(v: Value[_])(implicit ctx: EvaluationContext): Int = {
    val schemaMaybe = v.schema
    val hasSchema = schemaMaybe.isDefined
    val tokenType = v match {
      case t if KeyType.accepts(t) =>
        val key = v.asInstanceOf[KeyType.V]
        val ek = key.evaluate
        getKeyTokenType(ek.namespace.isDefined, key.attributes.isDefined)

      case t if StringType.accepts(t) =>
        val str = v.asInstanceOf[Value[StringType.T]].evaluate.toString
        val strBytes = str.getBytes(charset)
        val length = strBytes.length
        getStringTokenType(length)

      case t if BooleanType.accepts(t) =>
        val bool = v.asInstanceOf[Value[Boolean]].evaluate
        if (bool) DwTokenType.True else DwTokenType.False

      case t if NumberType.accepts(t) =>
        val number = v.asInstanceOf[Value[math.Number]].evaluate
        if (number.isWhole) {
          if (number.withinInt) {
            DwTokenType.Int
          } else if (number.withinLong) {
            DwTokenType.Long
          } else {
            DwTokenType.BigInt
          }
        } else if (number.withinDouble) {
          DwTokenType.Double
        } else {
          DwTokenType.BigDecimal
        }

      case t if NullType.accepts(t) =>
        DwTokenType.Null

      case t if RangeType.accepts(t) =>
        DwTokenType.Range

      case t if ArrayType.accepts(t) =>
        DwTokenType.ArrayStart

      case t if ObjectType.accepts(t) =>
        DwTokenType.ObjectStart

      case t if FunctionType.accepts(t) =>
        throw new DWBRuntimeExecutionException("Writing function values is not supported.")

      case t if RegexType.accepts(t) =>
        DwTokenType.Regex

      case t if DateTimeType.accepts(t) =>
        DwTokenType.DateTime

      case t if LocalDateTimeType.accepts(t) =>
        DwTokenType.LocalDateTime

      case t if LocalDateType.accepts(t) =>
        DwTokenType.LocalDate

      case t if TimeType.accepts(t) =>
        DwTokenType.Time

      case t if LocalTimeType.accepts(t) =>
        DwTokenType.LocalTime

      case t if TimeZoneType.accepts(t) =>
        DwTokenType.TimeZone

      case t if BinaryType.accepts(t) =>
        DwTokenType.Binary

      case t if PeriodType.accepts(t) =>
        DwTokenType.Period

      case t =>
        throw new DWBRuntimeExecutionException("Unexpected value type '" + t.toString + "'")

    }
    val hasSchemaProp = booleanToInt(hasSchema) << DwTokenHelper.SCHEMA_FLAG_RIGHT_BITS_BYTE
    hasSchemaProp | tokenType
  }
}

class WeaveBinaryWriterSettings() extends ConfigurableBufferSize with ConfigurableDeferred {
  var writeIndex: Boolean = _

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions() ++
      Map(BooleanModuleOption("writeIndex", defaultValue = false, descriptionUrl = "data-format/dwb/writeIndex.asciidoc"))
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "writeIndex" => {
        this.writeIndex = value.asInstanceOf[Boolean]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }

  }
}

class IndexHashSet[A] {
  private var index: Int = 0
  private val map = new mutable.LinkedHashMap[A, Int]()

  /**
    * Returns the index of the element, if it already existed the previous index is returned.
    */
  def put(obj: A): Int = {
    val maybeIndex = map.get(obj)
    if (maybeIndex.isEmpty) {
      val elemIndex = index
      map.put(obj, elemIndex)
      index = index + 1
      elemIndex
    } else {
      maybeIndex.get
    }
  }

  def indexOf(obj: A): Int = {
    val maybeIndex = map.get(obj)
    if (maybeIndex.isDefined) {
      maybeIndex.get
    } else {
      -1
    }
  }

  def contains(obj: A): Boolean = {
    map.contains(obj)
  }

  def iterator: Iterator[A] = {
    map.keysIterator
  }

  def size: Int = map.size
}
