package org.mule.weave.v2.module.dwb.reader

import org.mule.weave.v2.core.io.SeekableStream

import java.io.InputStream
import java.math.BigInteger
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.OffsetTime
import java.time.ZoneId
import java.time.ZonedDateTime
import java.time.temporal.TemporalAmount
import java.util
import java.util.Collections
import java.util.{ List => JList }
import java.util.{ Map => JMap }

import org.mule.weave.v2.dwb.api.IWeaveValue
import org.mule.weave.v2.dwb.api.IWeaveValueVisitor
import org.mule.weave.v2.dwb.api.values.IWeaveArrayValue
import org.mule.weave.v2.dwb.api.values.IWeaveBigDecimalValue
import org.mule.weave.v2.dwb.api.values.IWeaveBigIntValue
import org.mule.weave.v2.dwb.api.values.IWeaveBinaryValue
import org.mule.weave.v2.dwb.api.values.IWeaveBooleanValue
import org.mule.weave.v2.dwb.api.values.IWeaveDateTimeValue
import org.mule.weave.v2.dwb.api.values.IWeaveDoubleValue
import org.mule.weave.v2.dwb.api.values.IWeaveIntValue
import org.mule.weave.v2.dwb.api.values.IWeaveLocalDateTimeValue
import org.mule.weave.v2.dwb.api.values.IWeaveLocalDateValue
import org.mule.weave.v2.dwb.api.values.IWeaveLocalTimeValue
import org.mule.weave.v2.dwb.api.values.IWeaveLongValue
import org.mule.weave.v2.dwb.api.values.IWeaveNullValue
import org.mule.weave.v2.dwb.api.values.IWeaveObjectValue
import org.mule.weave.v2.dwb.api.values.IWeavePeriodValue
import org.mule.weave.v2.dwb.api.values.IWeaveStringValue
import org.mule.weave.v2.dwb.api.values.IWeaveTimeValue
import org.mule.weave.v2.dwb.api.values.IWeaveTimeZoneValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.LocalDateType
import org.mule.weave.v2.model.types.LocalTimeType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.PeriodType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.TimeType
import org.mule.weave.v2.model.types.TimeZoneType
import org.mule.weave.v2.model.values.math.{ Number => SNumber }
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.reader.exceptions.DWBRuntimeExecutionException

object WeaveValue {

  /**
    * Wrap dw values into WeaveValues (the API ones)
    */
  def apply(value: Value[_])(implicit ctx: EvaluationContext): IWeaveValue[_] = {
    value match {
      case x if StringType.accepts(x) =>
        new WeaveStringValue(value.asInstanceOf[StringType.V])

      case x if NumberType.accepts(x) =>
        val numberValue = value.asInstanceOf[NumberType.V]
        processNumber(numberValue)

      case x if BooleanType.accepts(x) =>
        new WeaveBooleanValue(value.asInstanceOf[BooleanType.V])

      case x if ObjectType.accepts(x) =>
        new WeaveObjectValue(value.asInstanceOf[ObjectType.V])

      case x if ArrayType.accepts(x) =>
        new WeaveArrayValue(value.asInstanceOf[ArrayType.V])

      case x if BinaryType.accepts(x) =>
        new WeaveBinaryValue(value.asInstanceOf[Value[BinaryType.T]])

      case x if NullType.accepts(x) =>
        new WeaveNullValue(value.asInstanceOf[Value[Null]])

      case t if DateTimeType.accepts(t) =>
        new WeaveDateTimeValue(value.asInstanceOf[Value[DateTimeType.T]])

      case t if LocalDateType.accepts(t) =>
        new WeaveLocalDateValue(value.asInstanceOf[Value[LocalDateType.T]])

      case t if PeriodType.accepts(t) =>
        new WeavePeriodValue(value.asInstanceOf[Value[PeriodType.T]])

      case t if LocalDateTimeType.accepts(t) =>
        new WeaveLocalDateTimeValue(value.asInstanceOf[Value[LocalDateTimeType.T]])

      case t if TimeType.accepts(t) =>
        new WeaveTimeValue(value.asInstanceOf[Value[TimeType.T]])

      case t if LocalTimeType.accepts(t) =>
        new WeaveLocalTimeValue(value.asInstanceOf[Value[LocalTimeType.T]])

      case t if TimeZoneType.accepts(t) =>
        new WeaveTimeZoneValue(value.asInstanceOf[Value[TimeZoneType.T]])

      case _ =>
        throw new DWBRuntimeExecutionException("Can't handle type: " + value.valueType.toString)
    }
  }

  private def processNumber(numberValue: NumberType.V)(implicit ctx: EvaluationContext): IWeaveValue[_] = {
    val schemaMaybe = numberValue.schema
    schemaMaybe match {
      case Some(schema) =>
        //return the value specific to the class
        val maybeClass = schema.valueOf("class")
        maybeClass match {
          case Some(classPropValue) =>
            processNumberWithClass(numberValue, classPropValue)

          case None =>
            processNumberWithoutClass(numberValue)
        }

      case None =>
        processNumberWithoutClass(numberValue)
    }
  }

  private def processNumberWithClass(numberValue: NumberType.V, classPropValue: Value[_])(implicit ctx: EvaluationContext): IWeaveValue[_] = {
    val className = StringType.coerce(classPropValue).evaluate.toString
    if (NumberPrecisionHelper.isInt(className)) {
      new WeaveIntValue(numberValue)
    } else if (NumberPrecisionHelper.isLong(className)) {
      new WeaveLongValue(numberValue)
    } else if (NumberPrecisionHelper.isDouble(className)) {
      new WeaveDoubleValue(numberValue)
    } else if (NumberPrecisionHelper.isBigInt(className)) {
      new WeaveBigIntValue(numberValue)
    } else if (NumberPrecisionHelper.isBigDecimal(className)) {
      new WeaveBigDecimalValue(numberValue)
    } else {
      val message = "Number class '" + className + "' not present in data-weave binary format"
      throw new DWBRuntimeExecutionException(message, classPropValue.location())
    }
  }

  private def processNumberWithoutClass(numberValue: NumberType.V)(implicit ctx: EvaluationContext): IWeaveValue[_] = {
    val number = numberValue.evaluate
    if (number.isWhole) {
      if (number.withinInt) {
        new WeaveIntValue(numberValue)
      } else if (number.withinLong) {
        new WeaveLongValue(numberValue)
      } else {
        new WeaveBigIntValue(numberValue)
      }
    } else if (number.withinDouble) {
      new WeaveDoubleValue(numberValue)
    } else {
      new WeaveBigDecimalValue(numberValue)
    }
  }

  def getSchema(value: Value[_])(implicit ctx: EvaluationContext): JMap[String, IWeaveValue[_]] = {
    value.schema match {
      case Some(schema) =>
        toWeaveValueMap(schema)

      case None =>
        Collections.emptyMap()
    }
  }

  def toWeaveValueMap(schema: Schema)(implicit ctx: EvaluationContext): JMap[String, IWeaveValue[_]] = {
    val map = new util.LinkedHashMap[String, IWeaveValue[_]]()
    val props = schema.properties()
    for (prop <- props) {
      val propName = prop.name.evaluate
      val propValue = WeaveValue(prop.value)
      map.put(propName, propValue)
    }
    map
  }
}

class WeaveStringValue(value: Value[CharSequence])(implicit ctx: EvaluationContext) extends IWeaveStringValue {

  def evaluate: String = {
    value.evaluate.toString
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitString(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(value)
  }

  override def toString: String = {
    '"' + evaluate + '"'
  }
}

class WeaveBooleanValue(value: Value[Boolean])(implicit ctx: EvaluationContext) extends IWeaveBooleanValue {

  def evaluate: java.lang.Boolean = {
    value.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitBoolean(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(value)
  }

  override def toString: String = {
    evaluate.toString
  }
}

class WeaveIntValue(number: Value[SNumber])(implicit ctx: EvaluationContext) extends IWeaveIntValue {

  def evaluate: Integer = {
    number.evaluate.toInt
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitInt(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(number)
  }

  override def toString: String = evaluate.toString
}

class WeaveDoubleValue(number: Value[SNumber])(implicit ctx: EvaluationContext) extends IWeaveDoubleValue {

  def evaluate: java.lang.Double = {
    number.evaluate.toDouble
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitDouble(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(number)
  }

  override def toString: String = evaluate.toString
}

class WeaveLongValue(number: Value[SNumber])(implicit ctx: EvaluationContext) extends IWeaveLongValue {

  def evaluate: java.lang.Long = {
    number.evaluate.toLong
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitLong(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(number)
  }

  override def toString: String = evaluate.toString
}

class WeaveBigIntValue(number: Value[SNumber])(implicit ctx: EvaluationContext) extends IWeaveBigIntValue {

  def evaluate: BigInteger = {
    number.evaluate.toBigInt.bigInteger
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(number)
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitBigInt(this)
  }

  override def toString: String = evaluate.toString
}

class WeaveBigDecimalValue(number: Value[SNumber])(implicit ctx: EvaluationContext) extends IWeaveBigDecimalValue {

  def evaluate: java.math.BigDecimal = {
    number.evaluate.toBigDecimal.bigDecimal
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitBigDecimal(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(number)
  }

  override def toString: String = evaluate.toString
}

class WeaveBinaryValue(binary: Value[SeekableStream])(implicit ctx: EvaluationContext) extends IWeaveBinaryValue {

  override def evaluate(): InputStream = {
    binary.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitBinary(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(binary)
  }

  override def toString: String = evaluate().toString
}

class WeaveNullValue(nullValue: Value[Null])(implicit ctx: EvaluationContext) extends IWeaveNullValue {

  override def evaluate(): Void = {
    nullValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitNull(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(nullValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveDateTimeValue(dateTimeValue: Value[ZonedDateTime])(implicit ctx: EvaluationContext) extends IWeaveDateTimeValue {

  override def evaluate(): ZonedDateTime = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitDateTime(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveLocalDateTimeValue(dateTimeValue: Value[LocalDateTime])(implicit ctx: EvaluationContext) extends IWeaveLocalDateTimeValue {

  override def evaluate(): LocalDateTime = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitLocalDateTime(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeavePeriodValue(period: Value[TemporalAmount])(implicit ctx: EvaluationContext) extends IWeavePeriodValue {

  override def evaluate(): TemporalAmount = {
    period.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitPeriod(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(period)
  }

  override def toString: String = evaluate().toString
}

class WeaveTimeValue(dateTimeValue: Value[OffsetTime])(implicit ctx: EvaluationContext) extends IWeaveTimeValue {

  override def evaluate(): OffsetTime = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitTime(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveLocalTimeValue(dateTimeValue: Value[LocalTime])(implicit ctx: EvaluationContext) extends IWeaveLocalTimeValue {

  override def evaluate(): LocalTime = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitLocalTime(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveTimeZoneValue(dateTimeValue: Value[ZoneId])(implicit ctx: EvaluationContext) extends IWeaveTimeZoneValue {

  override def evaluate(): ZoneId = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitTimeZone(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveLocalDateValue(dateTimeValue: Value[LocalDate])(implicit ctx: EvaluationContext) extends IWeaveLocalDateValue {

  override def evaluate(): LocalDate = {
    dateTimeValue.evaluate
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitLocalDate(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(dateTimeValue)
  }

  override def toString: String = evaluate().toString
}

class WeaveObjectValue(v: Value[ObjectSeq])(implicit ctx: EvaluationContext) extends IWeaveObjectValue {

  def evaluate: JMap[String, IWeaveValue[_]] = {
    val objectSeq: ObjectSeq = v.evaluate
    getProcessor() match {
      case Some(processor) =>
        new RedefinedObjectSeqMap(objectSeq, processor, getSchema(), v)

      case None =>
        new SimpleObjectSeqMap(objectSeq)
    }
  }

  private def getProcessor(): Option[String] = {
    for (
      schema <- v.schema;
      processor <- schema.valueOf("processor")
    ) yield {
      StringType.coerce(processor).evaluate.toString
    }
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitObject(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(v)
  }

  override def isObject: Boolean = true
}

class WeaveArrayValue(v: Value[ArraySeq])(implicit ctx: EvaluationContext) extends IWeaveArrayValue {

  def evaluate: JList[IWeaveValue[_]] = {
    val arraySeq: ArraySeq = v.evaluate
    new SimpleArraySeqList(arraySeq)
  }

  override def accept(visitor: IWeaveValueVisitor): Unit = {
    visitor.visitArray(this)
  }

  override def getSchema: JMap[String, IWeaveValue[_]] = {
    WeaveValue.getSchema(v)
  }

  override def isArray: Boolean = true
}

