package org.mule.weave.v2.module.dwb

import org.mule.weave.v2.module.core.common.TokenFactory
import org.mule.weave.v2.module.core.common.TokenHelper
import org.mule.weave.v2.module.core.common.TokenHelper.INVALID_TOKEN_ERROR_MSG
import org.mule.weave.v2.module.dwb.DwTokenType.DwTokenType
import org.mule.weave.v2.module.core.xml.reader.indexed.TokenArray
import org.mule.weave.v2.module.core.xml.reader.indexed.XmlTokenHelper.lengthToMask

/**
  * Bits distribution:
  * <ul>
  * <li> 40 bits -> Starting offset</li>
  * <li> 12 bits -> Name index</li>
  * <li> 12 bits -> Nesting depth</li>
  *
  * <li>  5 bits -> Token type</li>
  * <li> 16 bits -> Hash</li>
  * <li> 12 bits -> Namespace index</li>
  * <li> 31 bits -> Qname length</li>
  * </ul>
  */
object WeaveKeyToken {

  import org.mule.weave.v2.module.dwb.DwTokenHelper._

  def apply(offset: Long, tokenType: DwTokenType, depth: Int, length: Long, nameHash: Int, nameIndex: Int, nsIndex: Int = NO_NAMESPACE): Array[Long] = {
    assert(offset >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Offset `${offset}` is negative")
    assert(length >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Length `${length}` is negative")
    assert(depth >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Depth `${depth}` is negative")

    assert(offset < MAX_OFFSET, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Offset `${offset}` is bigger than allowed: ${MAX_OFFSET}")
    assert(length < MAX_QNAME_LEN, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Length `${length}` is bigger than allowed: ${MAX_QNAME_LEN}")
    assert(depth < MAX_DEPTH, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Depth `${depth}` is bigger than allowed: ${MAX_DEPTH}")

    val firstLong: Long = {
      val shiftedOffset: Long = offset << OFFSET_RIGHT_BITS
      val nameIndexLong = (nameIndex & MAX_NAME_INDEX) << NAME_INDEX_RIGHT_BITS
      shiftedOffset | nameIndexLong | depth
    }
    val secondLong: Long = {
      val tokenTypeLong = (tokenType & MAX_TOKEN_TYPE) << TOKEN_TYPE_RIGHT_BITS
      val hashLong = (nameHash & MAX_HASH) << HASH_RIGHT_BITS
      val nsIndexLong = (nsIndex & MAX_NS_INDEX) << NS_INDEX_RIGHT_BITS
      val lengthLong = length & MAX_QNAME_LEN
      tokenTypeLong | hashLong | nsIndexLong | lengthLong
    }
    TokenFactory.create(firstLong, secondLong)
  }
}

/**
  * Bits distribution:
  * <ul>
  * <li> 40 bits -> Starting offset</li>
  * <li> 12 bits -> Reserved</li>
  * <li> 12 bits -> Nesting depth</li>
  *
  * <li>  5 bits -> Token type</li>
  * <li>  1 bits -> Schema Flag </li>
  * <li> 58 bits -> Length</li>
  * </ul>
  */
object WeaveValueToken {

  import org.mule.weave.v2.module.dwb.DwTokenHelper._

  def apply(offset: Long, tokenType: DwTokenType, depth: Int, length: Long, hasSchema: Boolean): Array[Long] = {
    assert(offset >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Offset `${offset}` is negative")
    assert(depth >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Depth `${depth}` is negative")
    assert(length >= 0, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Length `${length}` is negative")

    assert(offset < MAX_OFFSET, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Offset `${offset}` is bigger than allowed: ${MAX_OFFSET}")
    assert(depth < MAX_DEPTH, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Depth `${depth}` is bigger than allowed: ${MAX_DEPTH}")
    assert(length < MASK_FULL_LEN, s"$INVALID_TOKEN_ERROR_MSG. DWB Token Length `${length}` is bigger than allowed: ${MASK_FULL_LEN}")

    val firstLong: Long = {
      val shiftedOffset: Long = offset << OFFSET_RIGHT_BITS
      shiftedOffset | depth
    }
    val secondLong: Long = {
      val tokenTypeLong = (tokenType & MAX_TOKEN_TYPE) << TOKEN_TYPE_RIGHT_BITS
      val schemaFlagLong = if (hasSchema) 0x1L << SCHEMA_FLAG_RIGHT_BITS_LONG else 0L
      val lengthLong = length & MASK_FULL_LEN
      tokenTypeLong | schemaFlagLong | lengthLong
    }
    TokenFactory.create(firstLong, secondLong)
  }
}

class DwTokenWrapper(token: Array[Long]) {
  import DwTokenHelper._

  private val offset = getOffset(token)
  private val depth = getDepth(token)
  private val tokenType = getTokenType(token)
  private val hasSchema = hasSchemaProps(token)
  private val typeStr = DwTokenType.getNameFor(tokenType)
  private val nameIndex = getNameIndex(token)
  private val length = getValueLength(token)

  override def toString = {
    s"offset=$offset, depth=$depth, tokenType=$tokenType, typeStr=$typeStr" +
      (if (hasSchema) " (with schema)" else "")

  }

}

object DwTokenHelper extends TokenHelper {
  //Common token attr lengths
  val OFFSET_LEN = 40
  val NAME_INDEX_LEN = 12
  val DEPTH_LEN = 12
  val TOKEN_TYPE_LEN = 5

  //Token lengths with Namespace
  val HASH_LEN = 16
  val NS_INDEX_LEN = 12
  val QNAME_LEN = 30

  //Token lengths without Namespace
  val SCHEMA_FLAG_LEN = 1
  val LENGTH_LEN = 58

  //offsets inside long (primitive) representation
  val OFFSET_RIGHT_BITS: Int = DEPTH_LEN + NAME_INDEX_LEN
  val NAME_INDEX_RIGHT_BITS: Int = DEPTH_LEN
  val DEPTH_RIGHT_BITS: Int = 0

  val TOKEN_TYPE_RIGHT_BITS: Int = LENGTH_LEN + SCHEMA_FLAG_LEN
  val SCHEMA_FLAG_RIGHT_BITS_LONG: Int = LENGTH_LEN
  val SCHEMA_FLAG_RIGHT_BITS_BYTE: Int = TOKEN_TYPE_LEN
  val HASH_RIGHT_BITS: Int = QNAME_LEN + NS_INDEX_LEN
  val NS_INDEX_RIGHT_BITS: Int = QNAME_LEN

  //max values
  val MAX_OFFSET: Long = lengthToMask(OFFSET_LEN)
  val MAX_NAME_INDEX: Long = lengthToMask(NAME_INDEX_LEN)
  val MAX_DEPTH: Long = lengthToMask(DEPTH_LEN)
  val MAX_TOKEN_TYPE: Long = lengthToMask(TOKEN_TYPE_LEN)
  val MAX_HASH: Long = lengthToMask(HASH_LEN)
  val MAX_NS_INDEX: Long = lengthToMask(NS_INDEX_LEN)
  val MAX_QNAME_LEN: Long = lengthToMask(QNAME_LEN)

  //Common masks
  val MASK_OFFSET: Long = MAX_OFFSET << OFFSET_RIGHT_BITS
  val MASK_NAME_INDEX: Long = MAX_NS_INDEX << NAME_INDEX_RIGHT_BITS
  val MASK_DEPTH: Long = MAX_DEPTH << DEPTH_RIGHT_BITS
  val MASK_TOKEN_TYPE: Long = MAX_TOKEN_TYPE << TOKEN_TYPE_RIGHT_BITS
  val MASK_SCHEMA_FLAG_LONG: Long = 0x1L << SCHEMA_FLAG_RIGHT_BITS_LONG
  val MASK_SCHEMA_FLAG_BYTE: Int = 0x1 << SCHEMA_FLAG_RIGHT_BITS_BYTE

  //With namespace masks
  val MASK_HASH: Long = MAX_HASH << HASH_RIGHT_BITS
  val MASK_NS_INDEX: Long = MAX_NS_INDEX << NS_INDEX_RIGHT_BITS

  val MASK_QNAME_LEN: Long = MAX_QNAME_LEN

  //without namespace mask
  val MASK_FULL_LEN: Long = lengthToMask(LENGTH_LEN)
  val NO_NAMESPACE: Int = MAX_NS_INDEX.toInt
  val NO_OFFSET_HASH_MASK: Int = MAX_HASH.toInt

  val TOKEN_TYPE_MASK: Int = lengthToMask(TOKEN_TYPE_LEN).toInt

  def hash(str: String): Int = {
    str.hashCode & NO_OFFSET_HASH_MASK
  }

  override def getDepth(longs: Array[Long]): Int = {
    ((longs(0) & MASK_DEPTH) >>> DEPTH_RIGHT_BITS).toInt
  }

  override def shouldAddToLocationCache(longs: Array[Long]): Boolean = {
    val tokenType = getTokenType(longs)
    tokenType == DwTokenType.ArrayStart || tokenType == DwTokenType.ObjectStart
  }

  def getTokenType(dwTokenWithFlags: Int): DwTokenType = {
    dwTokenWithFlags & DwTokenHelper.TOKEN_TYPE_MASK
  }

  override def getTokenType(longs: Array[Long]): DwTokenType = {
    getInt(longs(1), MASK_TOKEN_TYPE, TOKEN_TYPE_RIGHT_BITS)
  }

  def hasNamespace(dwToken: Int): Boolean = {
    dwToken == DwTokenType.KeyWithNS || dwToken == DwTokenType.KeyWithNSAttr
  }

  def hasAttributes(dwToken: Int): Boolean = {
    dwToken == DwTokenType.KeyWithAttr || dwToken == DwTokenType.KeyWithNSAttr
  }

  def hasSchemaProps(dwTokenWithFlags: Int): Boolean = {
    ((dwTokenWithFlags & MASK_SCHEMA_FLAG_BYTE) >>> SCHEMA_FLAG_RIGHT_BITS_BYTE) == 1
  }

  def hasSchemaProps(longs: Array[Long]): Boolean = {
    val i = getInt(longs(1), MASK_SCHEMA_FLAG_LONG, SCHEMA_FLAG_RIGHT_BITS_LONG)
    i == 1
  }

  override def getNameHash(longs: Array[Long]): DwTokenType = {
    getInt(longs(1), MASK_HASH, HASH_RIGHT_BITS)
  }

  override def getKeyLength(longs: Array[Long]): Long = {
    longs(1) & MASK_QNAME_LEN
  }

  override def getValueLength(longs: Array[Long]): Long = {
    longs(1) & MASK_FULL_LEN
  }

  override def getOffset(longs: Array[Long]): Long = {
    getInt(longs(0), MASK_OFFSET, OFFSET_RIGHT_BITS)
  }

  def getNameIndex(longs: Array[Long]): Int = {
    getInt(longs(0), MASK_NAME_INDEX, NAME_INDEX_RIGHT_BITS)
  }

  def getNsIndex(longs: Array[Long]): Option[Int] = {
    val nsIndex = getInt(longs(1), MASK_NS_INDEX, NS_INDEX_RIGHT_BITS)
    if (nsIndex < NO_NAMESPACE) {
      Some(nsIndex)
    } else {
      None
    }
  }

  def isKey(token: Array[Long]): Boolean = {
    val t = getTokenType(token)
    isKey(t)
  }

  def isKey(t: DwTokenType): Boolean = {
    t == DwTokenType.Key ||
      t == DwTokenType.KeyWithAttr ||
      t == DwTokenType.KeyWithNS ||
      t == DwTokenType.KeyWithNSAttr
  }

  /**
    * Applies a mask to get an int from some bits in the long
    */
  private def getInt(long: Long, mask: Long, rightBits: Int): Int = {
    getLong(long, mask, rightBits).toInt
  }

  /**
    * Applies a mask to get a long from some bits in the long
    */
  private def getLong(long: Long, mask: Long, rightBits: Int): Long = {
    (long & mask) >>> rightBits
  }

  def toHumanReadable(tokenArray: TokenArray): Seq[DwTokenWrapper] = tokenArray.map(new DwTokenWrapper(_))
}
