%dw 2.0

import fail from dw::Runtime
import * from dw::extension::DataFormat
import readLinesWith, writeLinesWith from dw::core::Binaries

/**
* Reads the Event Stream format and returns an Array of Objects where each object represents an Event
*/
fun readEventStream(content: Binary, charset: String, settings: Object): Array<Object> = do {
  content readLinesWith charset map ((line, index) -> do {
      if (line startsWith ":") null else do {
        var colon_index = line indexOf ":"
        ---
        if (colon_index > 1)
          {
            (line[0 to colon_index - 1]): trim(line[colon_index + 1 to -1])
          }
        else if (!isBlank(line))
          {
            (line): ""
          }
        else
          {}
      }
    }) then ((result) -> buildStreamEvents(result))
}

/**
* This function receives a list of objects and null, where each object only has one key value pair, and groups the objects until
* it reaches a Null value where it will start a new Object. So something like
* [{a: "1"}, {b:"2"}, null, {a: "3"}, null] -> [{a:"1",b:"2"}, {a:"3"}]
**/
fun buildStreamEvents(stream: Array<Object | Null>, currentEvent: Object = {}): Array<Object> =
  stream match {
    case [x ~ xs] -> do {
      x match {
        case is Null -> buildStreamEvents(xs, currentEvent) // This case is a comment
        case entry is Object -> do {
          if (isEmpty(entry))
            do {
              if (isEmpty(currentEvent)) // Ignore empty lines
                buildStreamEvents(xs)
              else
                [currentEvent ~ buildStreamEvents(xs)]
            }
          else
            buildStreamEvents(xs, currentEvent ++ entry)
        }
      }
    }
    case [] -> []
  }

/**
* Gets the list of events to write and write it with the SSE format
**/
fun writeEventStream(content: Array<{ _: Any }>, settings: Object): Binary = do {
  var streamLines = content match {
    case content is Array<{ _: Any }> -> do {
      content map ((item, index) -> do {
          item mapObject ((value, key, index) -> do {
              if (value is StringCoerceable)
                {
                  (key): value as String
                }
              else
                fail("Expecting type: `String` but got: `$(typeOf(value))` on key: `$(key)` at index: `$(index)`.")
            }) // Escape the new lines in the event data with spaces... is this the correct thing to do?
          pluck ((value, key, index) -> "$(key): $(value replace '\n' with ' ')") joinBy "\n" ++ "\n"
        })
    }
    else -> do {
      fail("Expecting input to be of type: `Array<{_: String}>` but got: `$(typeOf($))`")
    }
  }
  ---
  streamLines writeLinesWith "UTF-8"
}

@DataFormatExtension
var eventstream = {
  acceptedMimeTypes: ["text/event-stream"],
  fileExtensions: [".sse", ".eventstream"],
  label: "Event Stream",
  reader: readEventStream,
  writer: writeEventStream
}

