package org.mule.weave.v2.module.excel

import com.github.pjfanning.xlsx.SharedStringsImplementationType
import com.github.pjfanning.xlsx.StreamingReader

import java.io.File
import java.io.InputStream
import java.util.{ Iterator => JIterator }
import org.apache.poi.openxml4j.util.ZipSecureFile
import org.apache.poi.ss.usermodel._
import org.apache.poi.util.IOUtils
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.EmptyLocationCapable
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.excel.ExcelHelper.isEmptyRow
import org.mule.weave.v2.module.excel.StreamingCellUtils._
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider

import scala.annotation.tailrec
import scala.collection.mutable.LinkedHashMap

class ExcelStreamingReader(sourceProvider: SourceProvider, val settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends Reader {

  override def dataFormat: Option[DataFormat[_, _]] = Some(new ExcelDataFormat)

  private val inputStream = sourceProvider.asInputStream

  /**
    * Should return the root value of the document that was read
    *
    * @param name The name of the variable that is being referenced example payload
    * @return The root value
    */
  override protected def doRead(name: String): Value[_] = {
    if (settings.maxEntrySize.isDefined) {
      IOUtils.setByteArrayMaxOverride(settings.maxEntrySize.get)
    } else {
      IOUtils.setByteArrayMaxOverride(Integer.MAX_VALUE)
    }
    if (!settings.zipBombCheck) {
      ZipSecureFile.setMinInflateRatio(0)
    } else {
      if (settings.minInflateRatio.isDefined) {
        ZipSecureFile.setMinInflateRatio(settings.minInflateRatio.get)
      }
    }
    val workbook = StreamingReader.builder()
      .rowCacheSize(10) // number of rows to keep in memory (defaults to 10)
      .bufferSize(4096) // buffer size to use when reading InputStream to file (defaults to 1024)
      .setSharedStringsImplementationType(SharedStringsImplementationType.POI_DEFAULT) // Using old SharedStringsImplementationType to avoid furigana support
      .open(inputStream); // InputStream or File for XLSX file (required)
    ctx.registerCloseable(workbook)
    new IteratorObjectValue(new ExcelSheetIterator(workbook.sheetIterator(), settings), UnknownLocationCapable)
  }
}

object ExcelStreamingReader {

  def apply(file: File, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelStreamingReader = {
    new ExcelStreamingReader(SourceProvider(file), settings)
  }

  def apply(inputStream: InputStream, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelStreamingReader = {
    new ExcelStreamingReader(SourceProvider(inputStream), settings)
  }

  def apply(sourceProvider: SourceProvider, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelStreamingReader = {
    new ExcelStreamingReader(sourceProvider, settings)
  }

  def apply(content: String, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelStreamingReader = {
    new ExcelStreamingReader(SourceProvider(content), settings)
  }
}

class ExcelSheetIterator(sheetIterator: JIterator[Sheet], settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends Iterator[KeyValuePair] {

  override def hasNext: Boolean = {
    sheetIterator.hasNext
  }

  override def next(): KeyValuePair = {
    val sheet = sheetIterator.next()
    val sheetName = sheet.getSheetName

    KeyValuePair(KeyValue(sheetName), new ExcelStreamingSheetValue(sheet, settings))
  }

}

class ExcelStreamingSheetValue(sheet: Sheet, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends ArrayValue with EmptyLocationCapable {
  lazy val arraySeq = ArraySeq(new ExcelRowIterator(sheet.rowIterator(), settings))

  override def evaluate(implicit ctx: EvaluationContext): T = {
    arraySeq
  }
}

class ExcelRowIterator(rowIterator: JIterator[Row], settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends Iterator[Value[_]] {

  var nextMaybe: Option[Row] = None

  def headerRowNum(): Int = {
    settings.headerRowNum()
  }

  def headerColNum(headerRow: Row): Int = {
    if (settings.tableOffset.isDefined) settings.headerColNum() else headerRow.getFirstCellNum()
  }

  val headerRowMaybe: Option[Row] = {
    if (settings.header) {
      // This calculation positions the iterator in the first row (after the header)

      val settingsRowNum = headerRowNum()
      var firstRow = rowIterator.next()
      if (settings.tableOffset.isEmpty) {
        // has no start row number defined,
        // so use the first one the iterator returns
        settings._headerRowNum = firstRow.getRowNum
        Some(firstRow)
      } else {
        // skip until settingsRowNum
        while (firstRow.getRowNum < settingsRowNum) {
          firstRow = rowIterator.next()
        }
        if (firstRow.getRowNum == settingsRowNum) {
          Some(firstRow)
        } else {
          None
        }
      }
    } else {
      None
    }
  }

  val tableColNames: LinkedHashMap[Int, String] = {
    if (settings.header && headerRowMaybe.isDefined) {
      val map = new LinkedHashMap[Int, String]()
      val headerRow = headerRowMaybe.get
      val colNum = headerColNum(headerRow)
      settings._headerColNum = colNum
      for (i <- colNum until headerRow.getLastCellNum) {
        val cell = headerRow.getCell(i)
        if (cell != null) {
          map.+=((i, cellToString(cell)))
        } else {
          None
        }
      }
      map
    } else {
      new LinkedHashMap()
    }
  }

  private def fetchNext(): Option[Row] = {
    if (settings.ignoreEmptyLine) {
      fetchNextNonEmpty()
    } else {
      Some(rowIterator.next())
    }
  }

  @tailrec
  private def fetchNextNonEmpty(): Option[Row] = {
    if (!rowIterator.hasNext) {
      return None
    }
    val row = rowIterator.next()
    if (!isEmptyRow(row)) {
      return Some(row)
    }
    fetchNextNonEmpty()
  }

  override def hasNext: Boolean = {
    if (nextMaybe.isEmpty && rowIterator.hasNext) {
      nextMaybe = fetchNext()
    }
    nextMaybe.isDefined
  }

  private def popNext(): Option[Row] = {
    val next = nextMaybe
    nextMaybe = None
    next
  }

  override def next(): Value[_] = {
    if (hasNext) {
      val next = popNext()
      new ExcelRowValue(next, settings, tableColNames)
    } else {
      throw new IndexOutOfBoundsException("No more rows.")
    }
  }

}

object StreamingCellUtils extends CellUtils {
  protected def formulaToString(cell: Cell)(implicit ctx: EvaluationContext): String = {
    cell.getCellFormula
  }

  protected def formulaToValue(cell: Cell)(implicit ctx: EvaluationContext): StringValue = {
    StringValue(cell.getCellFormula)
  }
}